
/* Stream ADT:

   static Stream constant (int)
   static Stream intsFrom (int)
   static Stream cons (int, Stream)
   static Stream sum (Stream, Stream)
   static Stream filter (PredicateFunction, Stream);

   int head ()
   Stream tail ()

   SPECIFICATION:

   constant(i).head() = i
   intsFrom(i).head() = i
   cons(a,S).head() = a
   sum(S1,S2).head() = S1.head() + S2.head()

   constant(i).tail() = constant(i)
   intsFrom(i).tail() = intsFrom(i+1)
   cons(a,S).tail() = S
   sum(S1,S2).tail() = sum(S1.tail(),S2.tail())

*/



public abstract class Stream {

    public static Stream constant (int i) {
        return new StreamConstant(i);
    }

    public static Stream intsFrom (int i) {
        return new StreamIntsFrom(i);
    }

    static Stream cons (int i, Stream s) {
        return new StreamCons(i,s);
    }
        
    static Stream sum (Stream s1, Stream s2) {
        return new StreamSum(s1,s2);
    }

    static Stream filter (PredicateFunction pf, Stream s) {
        return new StreamFilter(pf,s);
    }

    static Stream sieve (Stream s) {
        return new StreamSieve(s);
    }

    public abstract int head ();
    public abstract Stream tail ();
}



class StreamConstant extends Stream {

    private int cnst;
    public StreamConstant (int i) {
        cnst = i;
    }

    public int head () {
        return cnst;
    }

    public Stream tail () {
        return Stream.constant(cnst);
    }

}



class StreamIntsFrom extends Stream {

    private int cnst;
    public StreamIntsFrom (int i) {
        cnst = i;
    }

    public int head () {
        return cnst;
    }

    public Stream tail () {
        return Stream.intsFrom(cnst+1);
    }

}



class StreamCons extends Stream {

    private int first;
    private Stream rest;

    public StreamCons (int i, Stream s) {
        first = i;
        rest = s;
    }

    public int head () {
        return first;
    }

    public Stream tail () {
        return rest;
    }

}



class StreamSum extends Stream {

    private Stream first;
    private Stream second;

    public StreamSum (Stream s1, Stream s2) {
        first = s1;
        second = s2;
    }

    public int head () {
        return first.head() + second.head();
    }

    public Stream tail () {
        return Stream.sum(first.tail(),second.tail());
    }

}



class StreamFilter extends Stream {

    private Stream underlying;
    private PredicateFunction predicate;

    public StreamFilter (PredicateFunction pf, Stream s) {
        underlying = s;
        predicate = pf;
    }

    public int head () {
        if (predicate.apply(underlying.head()))
            return underlying.head();
        else
            return Stream.filter(predicate,underlying.tail()).head();
    }

    public Stream tail () {
        if (predicate.apply(underlying.head()))
            return Stream.filter(predicate,underlying.tail());
        else
        return Stream.filter(predicate,underlying.tail()).tail();
    }

}



class StreamSieve extends Stream {

    private Stream stream;

    public StreamSieve (Stream s) {
        stream = s;
    }

    public int head () {
        return stream.head();
    }

    public Stream tail () {
        Stream multRemoved = 
	    Stream.filter(NotDivisibleBy.function(stream.head()),
			  stream.tail());
        return Stream.sieve(multRemoved);
    }
}
