(in-package "ACL2")

(include-book "records")
(include-book "seq")
(include-book "det-macros")

(defmacro seq-l1 (l1 &rest pairs)
  (seq-macro l1 (append (list :type ''latch1) pairs)))

(defmacro seq-l2 (l2 &rest pairs)
  (seq-macro l2 (append (list :type ''latch2) pairs)))

(defmacro seq-ma (MA &rest pairs)
  (seq-macro MA (append (list :type ''MA) pairs)))

(defun latch1 (validp op rc ra rb pch)
  (seq-l1 nil
          :validp  validp 
          :op      op 
          :rc      rc
          :ra      ra
          :rb      rb
          :pch     pch))

(defun latch2 (validp op rc ra-val rb-val pch)
  (seq-l2 nil
          :validp  validp 
          :op      op 
          :rc      rc
          :ra-val  ra-val
          :rb-val  rb-val
          :pch     pch))

(defun MA-state (pc regs imem dmem latch1 latch2)
  (seq-ma nil
          :pc      pc
          :regs    regs 
          :imem    imem
          :dmem    dmem
          :latch1  latch1
          :latch2  latch2))

(defun MA-p (x)
  (equal (g :type x) 'MA))

(defun ALU-output (op val1 val2)
  (cond ((equal op 'add) 
	 (+ val1 val2))
	((equal op 'sub) 
	 (- val1 val2))
	(t (* val1 val2))))

(defun in (x y)
  (if (endp y)
      nil
    (or (equal x (car y))
        (in x (cdr y)))))

(defun alu-opp (op)
  (in op '(add sub mul)))

(defun load-opp (op)
  (in op '(load loadi)))

(defun step-regs (MA)
  (let* ((regs     (g :regs MA))
	 (dmem     (g :dmem MA))
	 (latch2   (g :latch2 MA))
	 (validp   (g :validp latch2))
	 (op       (g :op latch2))
	 (rc       (g :rc latch2))
	 (ra-val   (g :ra-val latch2))
	 (rb-val   (g :rb-val latch2)))
    (if validp
	(cond ((alu-opp op)
	       (seq regs rc (ALU-output op ra-val rb-val)))
	      ((load-opp op)
	       (seq regs rc (g ra-val dmem)))
	      (t regs))
      regs)))

(defun rc-activep (op)
  (or (alu-opp op)
      (load-opp op)))

(defun uses-rbp (op)
  (or (alu-opp op)
      (in op '(store bez))))

(defun stall-l1p (MA)
  (let* ((latch1   (g :latch1 MA))
	 (l1validp (g :validp latch1))
	 (latch2   (g :latch2 MA))
	 (l1op     (g :op latch1))
	 (l2op     (g :op latch2))
	 (l2validp (g :validp latch2))
	 (l2rc     (g :rc latch2))
	 (l1ra     (g :ra latch1))
	 (l1rb     (g :rb latch1)))
    (and l2validp l1validp
	 (rc-activep l2op)
	 (or (equal l1ra l2rc)
             (and (uses-rbp l1op)
                  (equal l1rb l2rc))))))

(defun invalidate-l1p (MA)
  (let* ((latch1   (g :latch1 MA))
	 (l1op     (g :op latch1))
	 (l1validp (g :validp latch1))
	 (latch2   (g :latch2 MA))
	 (l2op     (g :op latch2))
	 (l2validp (g :validp latch2)))
    (or (and l1validp
             (in l1op '(bez jump)))
        (and l2validp
             (equal l2op 'bez)))))

(defun step-latch1 (MA)
  (let ((latch1 (g :latch1 MA))
	(inst   (g (g :pc MA) (g :imem MA))))
    (cond ((stall-l1p MA)
	   latch1)
	  ((invalidate-l1p MA)
	   (seq-l1 nil :validp nil))
	  (t (seq-l1 nil
                     :validp t
                     :op     (g :opcode inst)
                     :rc     (g :rc inst)
                     :ra     (g :ra inst)
                     :rb     (g :rb inst)
                     :pch    (g :pc MA))))))

(defun step-latch2 (MA)
  (let* ((latch1 (g :latch1 MA))
	 (l1op   (g :op latch1)))
    (if (or (not (g :validp latch1))
            (stall-l1p MA))
	(seq-l2 nil :validp nil)
      (seq-l2 nil
              :validp t
              :op     l1op
              :rc     (g :rc latch1)
              :ra-val (if (equal l1op 'load)
                          (g :ra latch1) 
                        (g (g :ra latch1) (g :regs MA)))
              :rb-val (g (g :rb latch1) (g :regs MA))
              :pch    (g :pch latch1)))))

(defun step-pc (MA)
  (let* ((pc       (g :pc MA))
	 (inst     (g (g :pc MA) (g :imem MA)))
	 (op       (g :opcode inst))
	 (regs     (g :regs MA))
	 (latch1   (g :latch1 MA))
         (l1op     (g :op latch1))
	 (latch2   (g :latch2 MA))
         (l2op     (g :op latch2))
	 (l2validp (g :validp latch2))
	 (l2ra-val (g :ra-val latch2))
	 (l2rb-val (g :rb-val latch2)))
    (cond ((stall-l1p MA)
	   pc)
	  ((invalidate-l1p MA)
	   (cond ((and l2validp
                       (equal l2op 'bez))
                  (if (equal 0 l2ra-val)
                      (ifix (ALU-output 'add pc l2rb-val))
                    (1+ pc)))
                 ((equal l1op 'jump)
                  (ifix (g (g :ra latch1) regs)))
                 (t pc)))
                  ;;; must be bez instruction
	  ((in op '(jump bez))
	   pc)
	  (t (1+ pc)))))

(defun step-dmem (MA)
  (let* ((dmem     (g :dmem MA))
	 (latch2   (g :latch2 MA))
	 (l2validp (g :validp latch2))
	 (l2op     (g :op latch2))
	 (l2ra-val (g :ra-val latch2))
	 (l2rb-val (g :rb-val latch2)))
    (if (and l2validp (equal l2op 'store))
	(seq dmem l2ra-val l2rb-val)
      dmem)))

(defun MA-step (MA)
  (seq-ma nil
          :pc     (step-pc MA)
          :regs   (step-regs MA)
          :dmem   (step-dmem MA)
          :imem   (g :imem MA)
          :latch1 (step-latch1 MA)
          :latch2 (step-latch2 MA)))

(defun MA-run (MA n)
  (if (zp n)
      MA
    (MA-run (MA-step MA) (1- n))))
