(in-package "ACL2")

(include-book "records")
(include-book "seq")

(defmacro seq-isa (ISA &rest pairs)
  (seq-macro ISA (append (list :type ''ISA) pairs)))

(defun ISA-state (pc regs imem dmem)
  (seq-isa nil 
           :pc   pc 
           :regs regs  
           :imem imem
           :dmem dmem))

(defun ISA-p (x)
  (equal (g :type x) 'ISA))

(defun add-rc (ra rb rc regs)
  (seq regs rc (+ (g ra regs)
		  (g rb regs))))

(defun ISA-add (rc ra rb ISA)
  (seq-isa nil
           :pc   (1+ (g :pc ISA))
           :regs (add-rc ra rb rc (g :regs ISA))
           :imem (g :imem ISA)
           :dmem (g :dmem ISA)))
	
(defun sub-rc (ra rb rc regs)
  (seq regs rc (- (g ra regs)
		  (g rb regs))))

(defun ISA-sub (rc ra rb ISA)
  (seq-isa nil 
           :pc   (1+ (g :pc ISA))
           :regs (sub-rc ra rb rc (g :regs ISA))
           :imem (g :imem ISA)
           :dmem (g :dmem ISA)))

(defun mul-rc (ra rb rc regs)
  (seq regs rc (* (g ra regs)
		  (g rb regs))))

(defun ISA-mul (rc ra rb ISA)
  (seq-isa nil
           :pc   (1+ (g :pc ISA))
           :regs (mul-rc ra rb rc (g :regs ISA))
           :imem (g :imem ISA)
           :dmem (g :dmem ISA)))

(defun load-rc (ad rc regs dmem)
  (seq regs rc (g ad dmem)))

(defun ISA-loadi (rc ra ISA)
  (let ((regs (g :regs ISA)))
    (seq-isa nil 
             :pc   (1+ (g :pc ISA))
             :regs (load-rc (g ra regs) rc regs (g :dmem ISA))
             :imem (g :imem ISA)
             :dmem (g :dmem ISA))))

(defun ISA-load (rc ad ISA)
  (seq-isa nil 
           :pc   (1+ (g :pc ISA))
           :regs (load-rc  ad rc (g :regs ISA) (g :dmem ISA))
           :imem (g :imem ISA)
           :dmem (g :dmem ISA)))

(defun store (ra rb regs dmem)
  (seq dmem (g ra regs) (g rb regs)))

(defun ISA-store (ra rb ISA)
  (seq-isa nil
           :pc   (1+ (g :pc ISA))
           :regs (g :regs ISA)
           :imem (g :imem ISA)
           :dmem (store ra rb (g :regs ISA) (g :dmem ISA))))

(defun bez (ra rb regs pc)
  (if (equal 0 (g ra regs))
      (ifix (+ pc (g rb regs)))
    (1+ pc)))

(defun ISA-bez (ra rb ISA)
  (seq-isa nil
           :pc (bez ra rb (g :regs ISA) (g :pc ISA))
           :regs (g :regs ISA)
           :imem (g :imem ISA)
           :dmem (g :dmem ISA)))

(defun ISA-jump (ra ISA)
  (seq-isa nil
           :pc (ifix (g ra (g :regs ISA)))
           :regs (g :regs ISA)
           :imem (g :imem ISA)
           :dmem (g :dmem ISA)))

(defun ISA-default (ISA)
  (seq-isa nil 
           :pc (1+ (g :pc ISA))
           :regs (g :regs ISA)
           :imem (g :imem ISA)
           :dmem (g :dmem ISA)))

(defun ISA-step (ISA)
  (let ((inst (g (g :pc ISA) (g :imem ISA))))
    (let ((op (g :opcode inst))
	  (rc (g :rc     inst))
	  (ra (g :ra     inst))
	  (rb (g :rb     inst)))
      (case op
	(add       (ISA-add rc ra rb ISA))  ; REGS[rc] := REGS[ra] + REGS[rb]
	(sub       (ISA-sub rc ra rb ISA))  ; REGS[rc] := REGS[ra] - REGS[rb]
	(mul       (ISA-mul rc ra rb ISA))  ; REGS[rc] := REGS[ra] * REGS[rb]
	(load      (ISA-load rc ra ISA))    ; REGS[rc] := MEM[ra]
	(loadi     (ISA-loadi rc ra ISA))   ; REGS[rc] := MEM[REGS[ra]]
	(store     (ISA-store ra rb ISA))   ; MEM[REGS[ra]] := REGS[rb]
	(bez       (ISA-bez ra rb ISA))     ; REGS[ra]=0 -> pc:=pc+REGS[rb]
	(jump      (ISA-jump ra ISA))       ; pc:=REGS[ra]
	(otherwise (ISA-default ISA))))))

(defun ISA-run (ISA n)
  (if (zp n)
      ISA
    (ISA-run (ISA-step ISA) (1- n))))
