package scg.game;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import scg.Constants;
import scg.Util;
import scg.game.BinaryGame.Result;
import scg.gen.Config;

public class Tournament {

    public static class Player {

        private final PlayerProxyI proxy;
        private int result;

        public Player(PlayerProxyI proxy) {
            this.proxy = proxy;
            this.result = 0;
        }

        public PlayerProxyI getProxy(){
            return proxy;
        }

        public int getResult(){
            return result;
        }

        public void addResultPoints(int points){
            result += points;
        }
    }

    private BinaryGame currentGame = null;
    private final Player[] players;
    private final Config config;

    /**  */
    public Tournament(Config config, PlayerProxyI... playerProxies) {
        this.config = config;
        this.players = new Player[playerProxies.length];
        for (int i = 0; i < playerProxies.length; i++) {
            this.players[i] = new Player(playerProxies[i]);
        }
    }

    /**  */
    public Tournament(Config config, edu.neu.ccs.demeterf.lib.List<PlayerProxyI> proxies) {
        this(config, proxies.toJavaList().toArray(new PlayerProxyI[0]));
    }

    /**  */
    public Tournament(Config config, List<? extends PlayerProxyI> playerProxies) {
        this(config, playerProxies.toArray(new PlayerProxyI[0]));
    }

    public Config getConfig(){
        return config;
    }

    private int gameNum = 0;

    public int getGameNum(){
        return gameNum;
    }

    private int numGames = 0;

    public int numGames(){
        return numGames;
    }

    public void start(HistoryFile tournamentHistory) throws Exception{
        for (int i = players.length - 1; i > 0; i--) {
            numGames += i + i;
        }

        String date = Util.getFileNameSafeDate(Util.now());
        for (int i = 0; i < players.length; i++) {
            for (int j = i + 1; j < players.length; j++) {
                startGame(tournamentHistory, i, j, date);
                startGame(tournamentHistory, j, i, date);
            }
        }
        tournamentHistory.tournamentResults(this);
        tournamentHistory.close();
    }

    public Player[] getResultSortedByPoints(){
        Player[] playersClone = Arrays.copyOf(players, players.length);
        Arrays.sort(playersClone, new Comparator<Player>() {

            public int compare(Player a, Player b){
                return b.getResult() - a.getResult();
            }
        });
        return playersClone;
    }

    private static String safeName(Player p){
        return p.getProxy().getSpec().getName().replace(' ', '_');
    }

    private int currentP1 = 0, currentP2 = 0;

    public String currentPlayer1(){
        return players[currentP1].getProxy().getSpec().getName();
    }

    public String currentPlayer2(){
        return players[currentP2].getProxy().getSpec().getName();
    }

    private void startGame(HistoryFile tournamentHistory, int i, int j, String date) throws Exception{
        gameNum++;
        currentP1 = i;
        currentP2 = j;
        currentGame = new BinaryGame(config, players[i].getProxy(), players[j].getProxy());
        Date start = Util.now();
        HistoryFile history = new HistoryFile("tournament", start, "-" + safeName(players[i]) + "-vs-"
                + safeName(players[j]) + Constants.HISTORY_FILE_SUFFIX);
        StatisticsCollector sc = new StatisticsCollector("tournament_statistic", start, "-" + safeName(players[i])
                + "-vs-" + safeName(players[j]) + Constants.HISTORY_FILE_SUFFIX);
        currentGame.start(history, sc);
        Result result = currentGame.getResult();
        players[i].addResultPoints(result.getPoints(players[i].getProxy().getName()));
        players[j].addResultPoints(result.getPoints(players[j].getProxy().getName()));
        tournamentHistory.recordCompetionEvent(result.toString());
    }

    @Override
    public String toString(){
        StringBuffer sb = new StringBuffer();
        sb.append("Player \t Points \n ============\n");
        for (Player player : getResultSortedByPoints()) {
            sb.append(player.getProxy().getSpec().getName() + "\t" + player.getResult() + "\n");
        }
        return sb.toString();
    }
}
