package scg.game;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import logging.Logger;
import scg.Util;
import scg.gen.*;
import edu.neu.ccs.demeterf.lib.Option;
import edu.neu.ccs.demeterf.lib.ident;

/**
 * Representation of AdminState
 * 
 * Constructed with: a set of playerSpecs, a configuration object holding all
 * configuration parameters such as: initial account balance, min decrement (for
 * reoffering)
 * 
 * Can be turned to a string but not necessarily parsed.
 * 
 * Can be queried for: the context for a specific player. spec of a specific
 * player. All players.
 * 
 * Provide methods for installing different player transactions.
 * 
 */
public class Game implements GameI {

    /** Version Identifier */
    public static final String REV = "$Rev: 551 $".substring(6, 9);

    // DONOTCHANGETHIS::

    /** Associates player id's to their proxy and state in the game */
    public static class PlayerStore {

        /** Player Id */
        private final int id;
        /** Proxy for contacting the player */
        private final PlayerProxyI proxy;
        /** Player's current Account */
        double account;
        /** Player's list of transactions */
        List<AccountTransaction> accountTransactions = new ArrayList<AccountTransaction>();
        /** Was this Player Kicked? */
        private boolean kicked = false;
        /** The reason the Player was Kicked */
        String kickReason = "";

        /** Return this Playe's ID Number */
        public int getId(){
            return id;
        }

        /** Return this Players Specification */
        public PlayerSpec getSpec(){
            return proxy.getSpec();
        }

        /** Return the Players Account value */
        public double getAccount(){
            return account;
        }

        /** Has the Player Been Kicked out of the Competition? */
        public boolean wasKicked(){
            return kicked;
        }

        /** Return the Number of Offered Challenges the Player has On the Market */
        public int numOffered(){
            return offeredChallenges.size();
        }

        /** Return the Number of Accepted Challenges the Player has On the Market */
        public int numAccepted(){
            return acceptedChallenges.size();
        }

        /** Return the Number of Provided Challenges the Player has On the Market */
        public int numProvided(){
            return providedChallenges.size();
        }

        /**
         * Challenges offered by the player The player shouldn't offer/reoffer
         * these.
         */
        Map<Integer, OfferedChallenge> offeredChallenges = new HashMap<Integer, OfferedChallenge>();
        /**
         * Challenges offered by the player and accepted by other player The
         * player has to provide these.
         */
        Map<Integer, AcceptedChallenge> acceptedChallenges = new HashMap<Integer, AcceptedChallenge>();
        /** Challenges provided for the player The player should solve these */
        Map<Integer, ProvidedChallenge> providedChallenges = new HashMap<Integer, ProvidedChallenge>();
        /** Challenges solved by other players */
        Map<Integer, SolvedChallenge> solvedChallenges = new HashMap<Integer, SolvedChallenge>();

        /** Create a Player Store with the initial Account */
        public PlayerStore(int playerId, PlayerProxyI playerProxy, double account) {
            this.id = playerId;
            this.proxy = playerProxy;
            this.account = account;
        }

        /** Take a Turn for the wrapped Player through the Proxy */
        public PlayerTrans takeTurn(PlayerContext currentPlayerContext) throws Exception{
            return proxy.takeTurn(currentPlayerContext);
        }

        /** Get the Reason the Player was Kicked */
        public String getKickReason(){
            return kickReason;
        }

        /** Set the Reason the Player was Kicked */
        public void setKickReason(String kr){
            kickReason = kr;
        }

        /** Get all account transactions */
        public List<AccountTransaction> getAccountTransactions(){
            return accountTransactions;
        }

        /** Add a transactions to the player's statement */
        public void addAccountTransaction(AccountTransaction accountTransaction){
            this.accountTransactions.add(accountTransaction);
        }

        /** Remove all account transactions */
        public void clearAccountTransaction(){
            this.accountTransactions.clear();
            // Remove the solved challenges too
            this.solvedChallenges.clear();
        }

        public void setKicked(boolean kicked){
            this.kicked = kicked;
        }

        public void setAccount(double account){
            this.account = account;
        }

    }

    private final Map<Integer, PlayerStore> playerStores = new HashMap<Integer, PlayerStore>();
    private final Map<Integer, Integer> challengeOfferer = new HashMap<Integer, Integer>();
    /** hold all secrets */
    Map<Integer, Solution> secrets = new HashMap<Integer, Solution>();
    private final Config config;

    /**  */
    public Game(Config config, PlayerProxyI... playerProxies) {
        this(config, Arrays.asList(playerProxies));
    }

    /**  */
    public Game(Config config, edu.neu.ccs.demeterf.lib.List<PlayerProxyI> proxies) {
        this(config, proxies.toJavaList());
    }

    /**  */
    public Game(Config config, List<? extends PlayerProxyI> playerProxies) {
        this.config = config;
        for (PlayerProxyI playerProxy : playerProxies) {
            int pid = config.createPlayerID();
            this.playerStores.put(pid, new PlayerStore(pid, playerProxy, config.getInitacc()));
        }
    }

    /** Current Competition Round */
    private int currentRound = 0;

    public synchronized int getCurrentRound(){
        return currentRound;
    }

    /** Current Player ID */
    private int currentPlayerID = 0;

    /** Turn Start */
    private long turnStart = System.currentTimeMillis();
    private HistoryFile history = null;

    /** Get The Start of the current turn */
    public long getTurnStart(){
        return turnStart;
    }

    /** Get the Game Configuration */
    public Config getConfig(){
        return config;
    }

    /** Return the ID of the player that is currently active */
    public synchronized String getCurrentPlayerName(){
        return currentPlayerID == 0 ? "" : playerStores.get(currentPlayerID).getSpec().getName();
    }

    public void start(HistoryFile history, StatisticsCollector sc) throws IOException{
        start(history, sc, false);
    }

    public void start(HistoryFile history, StatisticsCollector sc, boolean randomize) throws IOException{
        this.sc = sc;
        this.sc.startStatisticsCollection();
        start(history, randomize);
        this.sc.finishStatisticsCollection();
    }

    public void start(HistoryFile history, boolean randomize) throws IOException{
        this.history = history;
        Logger log = Logger.text(System.out, Util.logFileName("game"));
        int numPlayers = playerStores.size();
        history.header(getPlayers(playerStores));
        for (int round = 1; round <= config.getNumrounds() + config.getOtrounds() && numPlayers > 1; round++) {
            history.startRound(round);
        	if(config.isOverTime(round)){ 
        		history.recordEvent(-1, new OtherEvent(new ident("/* OverTime Round: Offers/Accepts/Reoffers are ignored */")));
        	}
            synchronized (this) {
                currentRound = round;
            }

            List<PlayerStore> stores = new ArrayList<PlayerStore>(playerStores.values());
            if (randomize) {
                Collections.shuffle(stores);
            }
            for (PlayerStore currentPlayerStore : stores) {
                if (numPlayers <= 1) {
                    break;
                }
                turnStart = System.currentTimeMillis();
                if (currentPlayerStore.wasKicked()) {
                    continue;
                }
                synchronized (this) {
                    currentPlayerID = currentPlayerStore.getId();
                }
                PlayerSpec currentPlayerSpec = currentPlayerStore.getSpec();
                // log.event("Player Turn: " + currentPlayerID + " : " +
                // currentPlayerSpec);
                PlayerContext currentPlayerContext = getPlayerContext(currentPlayerID, round);
                // start a fresh statement next time
                currentPlayerStore.clearAccountTransaction();
                try {
                    turnStart = System.currentTimeMillis();
                    PlayerTrans trans = currentPlayerStore.takeTurn(currentPlayerContext);
                    //log.notify("Turn Completed: "+Util.format((System.currentTimeMillis()-turnStart)/1000.0)+" sec");
                    synchronized (this) {
                        trans = preprocessTransaction(trans);
                    }
                    history.recordEvent(currentPlayerID, trans);
                    currentPlayerContext.isLegal(trans);
                    synchronized (this) {
                        trans = trans.applyTransactions(this, config.isOverTime(round));
                        // history.replaceEvent(currentPlayerID, trans);
                    }
                } catch (GameI.BadTransException ex) {
                    kick(currentPlayerID, "" + ex.getMessage(), history);
                    numPlayers--;
                    log.error("BadTransaction[" + currentPlayerSpec.getName() + "]: " + ex.getMessage() + "\n");
                } catch (GameI.HTTPTransException ex) {
                    kick(currentPlayerID, "" + Util.rootCause(ex).getMessage(), history);
                    numPlayers--;
                    log.error("(HTTP/Parse)Exception[" + currentPlayerSpec.getName() + "]: "
                            + Util.rootCause(ex).getMessage() + "\n");
                    ex.storeException(currentPlayerSpec.getName());
                } catch (Exception ex) {
                    // PlayerProxy issues/Connection Problems
                    Throwable t = Util.rootCause(ex);
                    kick(currentPlayerID, "" + t.getMessage(), history);
                    numPlayers--;
                    log.error("Exception[" + currentPlayerSpec.getName() + "]: " + t);
                    log.notify("StackTrace:");
                    for (StackTraceElement e : t.getStackTrace()) {
                        log.notify("    " + e.toString());
                    }
                }
            }
            numPlayers -= checkBalances(history);
            history.flushRound();
        }
        currentPlayerID = 0;
        history.footer(getPlayersSortedByAccountOrKicked());
        history.close();
    }

    private static final Solution EmptySolution = new Solution(edu.neu.ccs.demeterf.lib.ListMap.<Var, Boolean>create());

    private PlayerTrans preprocessTransaction(PlayerTrans trans){
        return new PlayerTrans(trans.getId(), trans.getTs().map(
                new edu.neu.ccs.demeterf.lib.List.Map<Transaction, Transaction>() {

                    @Override
                    public Transaction map(Transaction t){
                        if (t instanceof OfferTrans) {
                            OfferTrans ot = (OfferTrans) t;
                            // validateOffer(ot, config);
                            return new OfferTrans(config.createChallengeID(), ot.getKind(), ot.getPred(), ot.getPrice());
                        } else if (t instanceof ProvideTrans) {
                            ProvideTrans pt = (ProvideTrans) t;
                            int challengeId = pt.getChallengeid();
                            if (pt.getSecret().isSome()) {
                                Game.this.secrets.put(challengeId, pt.getSecret().inner());
                                // Sending an obfuscated solution so that the
                                // transaction remains legal. Sending a none
                                // solution will make the transaction illegal.
                                return new ProvideTrans(pt.getInst(), Option.some(EmptySolution), challengeId);
                            }
                        }
                        return t;
                    }
                }));
    }

    /**
     * Make sure the Offer is good public void validateOffer(OfferTrans ot,
     * Config config){ Predicate pred = config.getPredicate(); if
     * (!pred.valid(ot.getPred())) { throw new GameI.BadTransException("Illegal
     * Type Offered: " + ot.getPred()); } }
     */
    public List<PlayerStore> getPlayersSortedByAccountOrKicked(){
        List<PlayerStore> stores = getPlayers();
        Collections.sort(stores, new Comparator<PlayerStore>() {

            public int compare(PlayerStore a, PlayerStore b){
                if (a.wasKicked() && !b.wasKicked()) {
                    return 1;
                }
                if (!a.wasKicked() && b.wasKicked()) {
                    return -1;
                }
                if (a.account == b.account) {
                    return 0;
                }
                if (a.account < b.account) {
                    return 1;
                }
                return -1;
            }
        });
        return stores;
    }

    /** Returns a list of participants */
    public List<PlayerStore> getPlayers(){
        List<PlayerStore> stores = new ArrayList<PlayerStore>();
        stores.addAll(playerStores.values());
        return stores;
    }

    private int checkBalances(HistoryFile history){
        int numKicked = 0;
        for (PlayerStore ps : playerStores.values()) {
            if (!ps.wasKicked() && ps.account < 0) {
                kick(ps.getId(), "Negative balance", history);
                numKicked++;
            }
        }
        return numKicked;
    }

    /** Return a Collection of PlayerStores */
    public java.util.Collection<PlayerStore> getPlayerStores(){
        return playerStores.values();
    }

    // All challenges where the kicked player is the challenger must be
    // refunded.
    // All challenges where the kicked player is the challengee are just dropped
    // without refund.
    // player is removed from current players list.
    private void kick(int playerID, String errorMessage, HistoryFile history){
        for (Entry<Integer, PlayerStore> player : playerStores.entrySet()) {
            // challenges kicked player accepted from other players
            PlayerStore playerStore = player.getValue();
            for (AcceptedChallenge challenge : playerStore.acceptedChallenges.values()) {
                if (challenge.getChallengee().getId() == playerID) {
                    playerStore.acceptedChallenges.remove(challenge);
                    // no refund
                }
            }
            // challenges kicked player provided to other players
            for (Challenge challenge : playerStore.providedChallenges.values()) {
                if (challenge.getChallenger().getId() == playerID) {
                    playerStore.providedChallenges.remove(challenge);
                    // refund
                    AccountTransaction t = new AccountRefundTrans(challenge.getPrice(), challenge.getKey(), player
                            .getKey(), playerID);
                    addPlayerAccountTransaction(t);

                    // transferMoney(playerID, player.getKey(),
                    // challenge.getPrice());
                }
            }
            if (player.getKey() == playerID) {
                // challenges other players accepted from the kicked player
                for (AcceptedChallenge challenge : player.getValue().acceptedChallenges.values()) {
                    // refund
                    AccountTransaction t = new AccountRefundTrans(challenge.getPrice(), challenge.getKey(), challenge
                            .getChallenger().getId(), playerID);
                    addPlayerAccountTransaction(t);

                    // transferMoney(playerID,
                    // challenge.getChallenger().getId(), challenge.getPrice());
                }
                player.getValue().acceptedChallenges.clear();
                // challenges other players provided for the kicked player
                player.getValue().providedChallenges.clear();
                // All challenges offered by the kicked player but never
                // accepted
                player.getValue().offeredChallenges.clear();
                // player is added to the kicked players list.
                player.getValue().setKicked(true);
                // Set the reason
                player.getValue().setKickReason(errorMessage);
            }

        }
        history.recordEvent(playerID, new PlayerKickedEvent(playerStores.get(playerID).getSpec(), errorMessage, Util
                .printDate(Util.now())));

    }

    /**
     * Retrieve a list of players in a demeterf map to be printed in the history
     * file
     */
    private edu.neu.ccs.demeterf.lib.List<edu.neu.ccs.demeterf.lib.Entry<PlayerID, PlayerSpec>> getPlayers(
            Map<Integer, PlayerStore> playerStores){
        edu.neu.ccs.demeterf.lib.List<edu.neu.ccs.demeterf.lib.Entry<PlayerID, PlayerSpec>> players = edu.neu.ccs.demeterf.lib.List
                .create();
        for (PlayerStore ps : playerStores.values()) {
            players = players.push(new edu.neu.ccs.demeterf.lib.Entry<PlayerID, PlayerSpec>(new PlayerID(ps.getId()),
                    ps.getSpec()));
        }
        return players;
    }

    /** Retrieve the player context for the current game */
    public PlayerContext getPlayerContext(int playerID, int currentRound){
        PlayerStore store = playerStores.get(playerID);
        List<OfferedChallenge> otherOffered = getOtherOffers(playerID);

        // Commented out for now... Will use it later
        /*
         * PlayerContext playerContext = new PlayerContext(config, new
         * PlayerID(playerID), store.account, currentRound,
         * Util.toDemF(store.offeredChallenges.values()),
         * Util.toDemF(otherOffered), Util.toDemF(store.acceptedChallenges.values()),
         * Util.toDemF(store.providedChallenges.values()), Util.toDemF(store.getAccountTransactions()));
         */

        PlayerContext playerContext = new PlayerContext(config, new PlayerID(playerID), store.account, currentRound,
                Util.toDemF(store.offeredChallenges.values()), Util.toDemF(otherOffered),
                Util.toDemF(store.acceptedChallenges.values()), Util.toDemF(store.providedChallenges.values()),
                Util.toDemF(store.solvedChallenges.values()),
                edu.neu.ccs.demeterf.lib.Option.<AccountTransactionList> none());

        return playerContext;
    }

    public double calculatePaybackToAcceptor(double achievedQuality, double secretQuality, double price,
            double profitFactor){
        boolean acceptorWins = achievedQuality >= price * secretQuality;
        if (acceptorWins) {
            return price /* what was paid */+ profitFactor * achievedQuality;
            // double winAmount = achievedQuality - price * secretQuality;
            // return price /* what was paid */+ profitFactor * winAmount;
        } else {
            return 0;
        }
    }

    /** Collect offers made by other players */
    private List<OfferedChallenge> getOtherOffers(int playerID){
        List<OfferedChallenge> otherOffered = new ArrayList<OfferedChallenge>();
        for (Entry<Integer, PlayerStore> player : playerStores.entrySet()) {
            if (player.getKey() != playerID) {
                PlayerStore playerStore = player.getValue();
                otherOffered.addAll(playerStore.offeredChallenges.values());
            }
        }
        return otherOffered;
    }

    /** Transfer money between the accounts of two players */
    public void transferMoney(int fromPlayerId, int toPlayerId, double amount){
        playerStores.get(fromPlayerId).account -= amount;
        playerStores.get(toPlayerId).account += amount;
    }

    public void addPlayerAccountTransaction(AccountTransaction accountTrans){
        // record Event
        history.recordEvent(0, accountTrans);
        // Update account
        accountTrans.applyTransaction(this);
        // Add transaction to both player's statements
        playerStores.get(accountTrans.getAcceptor()).addAccountTransaction(accountTrans);
        playerStores.get(accountTrans.getOfferer()).addAccountTransaction(accountTrans);
    }

    // Install transaction methods for all transaction types */

    /** Handle an offered challenge transaction */
    public OfferTrans installTransaction(int challengerID, OfferTrans ot){
        OfferedChallenge challenge = new OfferedChallenge(ot.getChallengeid(), new PlayerID(challengerID),
                ot.getPred(), ot.getPrice(), ot.getKind());
        challengeOfferer.put(challenge.getKey(), challengerID);
        playerStores.get(challengerID).offeredChallenges.put(challenge.getKey(), challenge);
        return ot;
    }

    /** Handle an accept challenge transaction */
    public AcceptTrans installTransaction(int challengeeID, AcceptTrans at){
        int challengeID = at.getChallengeid();
        int challengerID = challengeOfferer.get(challengeID);
        PlayerStore challengerStore = playerStores.get(challengerID);
        OfferedChallenge toBeAcceptedChallenge = challengerStore.offeredChallenges.get(challengeID);

        AcceptedChallenge acceptedChallenge = toBeAcceptedChallenge.accept(new PlayerID(challengeeID));

        // Create an account accept transaction
        AccountAcceptTrans actTrans = new AccountAcceptTrans(acceptedChallenge.getPrice(), challengeID, challengeeID,
                challengerID);
        // Execute the account transaction
        addPlayerAccountTransaction(actTrans);

        // transferMoney(challengeeID, challengerID,
        // acceptedChallenge.getPrice());

        challengerStore.offeredChallenges.remove(challengeID);
        challengerStore.acceptedChallenges.put(challengeID, acceptedChallenge);
        return at;
    }

    /** Handle Provide Transactions */
    public ProvideTrans installTransaction(int challengerID, ProvideTrans pt){
        int challengeID = pt.getChallengeid();
        PlayerStore challengerStore = playerStores.get(challengerID);
        AcceptedChallenge toBeProvidedChallenge = challengerStore.acceptedChallenges.get(challengeID);
        int challengee = toBeProvidedChallenge.getChallengee().getId();

        Problem problem = pt.getInst();
        ProvidedChallenge providedChallenge = toBeProvidedChallenge.provide(problem);

        PlayerStore challengeeStore = playerStores.get(challengee);
        challengerStore.acceptedChallenges.remove(challengeID);
        challengeeStore.providedChallenges.put(challengeID, providedChallenge);
        return pt;
    }

    /** Handle Solve Transactions */
    public SolveTrans installTransaction(int challengeeID, SolveTrans st){
        int challengeID = st.getChallengeid();
        PlayerStore challengeeStore = playerStores.get(challengeeID);
        ProvidedChallenge toBeSolvedChallenge = challengeeStore.providedChallenges.get(challengeID);
        int challengerID = toBeSolvedChallenge.getChallenger().getId();

        Problem inst = toBeSolvedChallenge.getInstance();
        Solution sol = st.getSol();

        Objective objective = config.getObjective();
        double quality = config.getObjective().value(inst, sol);
        double secretQuality = 1.0;
        Solution secret = secrets.get(challengeID);
        if (secret != null) {
            secretQuality = objective.value(inst, secret);
            secrets.remove(challengeID);
            objective.value(inst, secret);
            try {
                history.secretRevealedEvent(challengeID, secret);
            } catch (IOException e) {
                System.err.println("History file closed");
            }
        }
        double price = toBeSolvedChallenge.getPrice();
        double profitFactor = config.getProfitFactor();
        double profit = calculatePaybackToAcceptor(quality, secretQuality, price, profitFactor);

        if (sc != null) {
            sc.collectStatistic(challengeID, playerStores.get(challengerID).getSpec().getName(), playerStores.get(
                    challengeeID).getSpec().getName(), toBeSolvedChallenge.getPred(), price, quality, secretQuality,
                    profit);
        }
        // create account transaction
        AccountSolveTrans slvTrans = new AccountSolveTrans(profit, quality, challengeID, challengeeID, challengerID);

        // Apply Transaction
        addPlayerAccountTransaction(slvTrans);

        // Update both player's accounts
        // transferMoney(challengerID, challengeeID, profit);

        challengeeStore.providedChallenges.remove(challengeID);
        PlayerStore challengerStore = playerStores.get(challengeeID);
        challengerStore.solvedChallenges.put(challengeID, toBeSolvedChallenge.solve(sol, quality));
        return st;
    }

    private StatisticsCollector sc = null;

    /** Handle Reoffer Transactions */
    public ReofferTrans installTransaction(int newChallenger, ReofferTrans rt){
        int challengeID = rt.getChallengeid();

        int challengerID = challengeOfferer.get(challengeID);

        challengeOfferer.remove(challengeID);
        challengeOfferer.put(challengeID, newChallenger);

        PlayerStore challengerStore = playerStores.get(challengerID);
        PlayerStore newChallengerStore = playerStores.get(newChallenger);

        OfferedChallenge toBeReofferedChallenge = challengerStore.offeredChallenges.get(challengeID);
        OfferedChallenge reOfferedChallenge = toBeReofferedChallenge
                .reoffer(new PlayerID(newChallenger), rt.getPrice());

        challengerStore.offeredChallenges.remove(challengeID);
        newChallengerStore.offeredChallenges.put(challengeID, reOfferedChallenge);
        return rt;
    }

    /** DGP method from Class PrintHeapToString */
    @Override
    public String toString(){
        return scg.gen.PrintHeapToString.PrintHeapToStringM(this);
    }
}
