package scg.game;

import edu.neu.ccs.demeterf.lib.*;
import edu.neu.ccs.demeterf.*;
import scg.gen.*;
import scg.Util;

/** Check all the Player Transaction Rules/Constraints */
public class CheckTrans extends StaticTP{
    /** Game Configuration */
    final Config config;
    /** Corresponding PlayerContext */
    final PlayerContext ctxt;
    /** Is this an overtime round? */
    final boolean overtime;
    
    static{
        // Check Transaction requires assertions!
        boolean assertActive = false;
        assert (assertActive = true);
        if(!assertActive){
            System.err.println("Check Transaction requires assertions to be enabled\n"+
                    "  Must pass \"-ea:scg.game.CheckTrans\" to the JVM");
            System.exit(1);
        }
    } 
    
    public CheckTrans(PlayerContext ctx){
        config = ctx.getConfig();
        ctxt = ctx;
        overtime = ctx.getCurrentRound() > config.getNumrounds();
    }
    
    /** Traversa the PlayerTransaction and check all the "rules" */
    public static void checkTransaction(PlayerTrans pt, PlayerContext ctx){
        new Traversal(new CheckTrans(ctx)).traverse(pt);
    }

    /** Return a challenge predicate that matches a challenge ID */
    <CT extends Challenge> List.Pred<CT>idMatch(final int challengeid){
        return new List.Pred<CT>(){
            public boolean huh(CT a){ return a.getKey() == challengeid; }
        };
    }
    
    /** Global/Whole-transaction validation checks */
    public PlayerTrans combine(PlayerTrans h, PlayerID id, List<Transaction> ts){
        assert id.equals(ctxt.getId()) : "Current Player must Respond: "+id+" != "+ctxt.getId();
        
        // Don't Accept and Reoffer the same challenge
        assert !h.getAcceptTrans().containsAnyG(h.getReofferTrans(), new List.GComp<AcceptTrans, ReofferTrans>() {
            public boolean comp(AcceptTrans a, ReofferTrans r){
                return a.getChallengeid() == r.getChallengeid();
            }
        }) : "Cannot accept and reoffer the same Challenge";

        // Solve all provided challenges
        assert ctxt.getProvided().sameG(h.getSolvedTrans(), new List.GComp<ProvidedChallenge, SolveTrans>(){
            public boolean comp(ProvidedChallenge c, SolveTrans t){
                return c.getKey() == t.getChallengeid();
            }
        }) : "Must solve all provided challenges";

        // Provide for all accepted challenges
        assert ctxt.getAccepted().sameG(h.getProvidedTrans(), new List.GComp<AcceptedChallenge, ProvideTrans>(){
            public boolean comp(AcceptedChallenge c, ProvideTrans t){
                return c.getKey() == t.getChallengeid();
            }
        }) : "Must provide for all accepted challenges";
        
        // Agents are active, but not too much so
        if(!overtime){
            int numProposals = h.getOfferTrans().length() + h.getReofferTrans().length();
            int numOppositions = h.getAcceptTrans().length() + h.getReofferTrans().length();
            int maxProposals = config.getMaxProposals();
            int minProposals = config.getMinProposals();
            int minOppositions = Math.min(config.getMinPropositions(), ctxt.getTheirOffered().length());

            assert (numProposals >= minProposals) : "Must propose at least "+minProposals;
            assert (numProposals <= maxProposals) : "Cannot propose more than "+maxProposals;
            assert (numOppositions >= minOppositions) : "Must oppose at least "+minOppositions;

            // Proposed enough secrets
            if(config.getHasSecrets()){
                List<OfferTrans> ots = h.getOfferTrans();
                int secrets = ots.count(new List.Pred<OfferTrans>(){
                    public boolean huh(OfferTrans t){ return t.getKind().isSecret(); }
                });
                assert secrets >= (int) (config.getSecretRatio() * ots.length() + (1.0 - config.getSecretRatio())) : 
                    "Must offer at least "+(config.getSecretRatio()*100)+"% secret challenges";
            }
        }
        return h;
     }
    
    /** Rules enforced for Offered Challenges/Transactions */
    public OfferTrans combine(OfferTrans h, int challengeid, ChallengeKind kind, final ProblemType pred, double price){
        assert 0 <= price && price <= 1 : "The price of a Challenge Must be in [0..1]: "+price;
        assert !kind.isSecret() || config.getHasSecrets() : "Secret challenges cannot be offered unless the configuration allows them";
        
        // Problem type must be distinct integers (implemented as a set...) 
        assert pred.getType().toList().andmap(new List.Pred<Integer>(){
            public boolean huh(Integer i){
                return (0 <= i && i <= 255); 
            }
        }) : "Problem Type must contain valid relation numbers: "+pred.getType();
     
        assert !ctxt.getAllOffered().contains(new List.Pred<OfferedChallenge>(){
            public boolean huh(OfferedChallenge oc){
                return oc.getPred().equals(pred);
            }
        }) : "The types of Offered Challenges must be new";

        Option<String> error = config.getPredicate().valid(pred);
        assert !error.isSome() : "Invalid offered type (" + error.inner() + "): " + pred;

        return h;
    }
    
    /** Rules enforced for Accepted Challenges/Transactions */
    public AcceptTrans combine(AcceptTrans h, final int challengeid){
        assert ctxt.getTheirOffered().contains(new List.Pred<OfferedChallenge>(){
            public boolean huh(OfferedChallenge ot){
                return ot.getKey() == challengeid;
            }
        }) : "Must Accept only other\'s challenges: "+challengeid;
        
        return h;
    }
    
    /** Rules enforced for Provided Challenges/Transactions */
    public ProvideTrans combine(ProvideTrans h, Problem inst, Option<Solution> secret, final int challengeid){
        List.Pred<AcceptedChallenge> ch = idMatch(challengeid);
        assert ctxt.getAccepted().contains(ch) : "Provide only Accepted Challenges: "+challengeid;
        
        AcceptedChallenge ac = ctxt.getAccepted().find(ch); 
        assert !ac.isSecret() || secret.isSome() : "Secret challenges must provide a Secret: "+challengeid;
        
        Option<String> error = config.getPredicate().valid(inst, ac.getPred());
        assert !error.isSome() : "Provided problem must match challenge type: "+challengeid;
        
        return h;
    }

    /** Rules enforced for Reoffered Challenges/Transactions */
    public ReofferTrans combine(ReofferTrans h, final int challengeid, double price){
        List.Pred<OfferedChallenge> ch = idMatch(challengeid);
        assert ctxt.getTheirOffered().contains(ch) : "Must Reoffer only other\'s challenges: "+challengeid;
        
        OfferedChallenge oc = ctxt.getTheirOffered().find(ch);
        assert oc.getPrice()-price > config.getMindecr()-Util.DELTA :
            "Reoffered challenge must be decremented at least "+config.getMindecr()+": "+challengeid;
        
        return h;
    }
    
    /** Rules enforced for Problems */
    public Problem combine(Problem h, List<Var> vars, Cons<Clause> clauses){
        Predicate pred = config.getPredicate();
        assert Problem.setOfVars(clauses).subseteq(Set.create(vars)) : "Set of declared variables must be a super set of the set of used variables";
        assert vars.length() <= pred.getMaxClauses() : "Number of declared variables is limited by the configuration: "+vars.length();
        assert clauses.length() <= pred.getMaxClauses() : "Number of clauses is limited by the configuration: "+clauses.length();
        
        return h;
    }
    public Clause combine(Clause h, int relnum, int weight, List<Var> vars){
        assert 0 <= relnum && relnum <= 255 : "Relation number must be in [0..255]: "+relnum;
        assert 0 < weight : "Weight must be a positive, non-zero integer: "+weight;
        assert vars.length() == 3 : "Clause Must contain 3 variables";

        return h;
    }
}
