package scg.game;

import java.io.IOException;
import java.util.List;

import scg.Util;
import scg.gen.Config;

public class BinaryGame extends Game {

    public static final int WIN_POINTS = 3;
    public static final int TIE_POINTS = 1;
    public static final int LOSS_POINTS = 0;

    private final Result result;

    public BinaryGame(Config config, PlayerProxyI player1, PlayerProxyI player2) {
        super(config, player1, player2);
        result = new Result(player1.getName(), player2.getName());
    }

    @Override
    public void start(HistoryFile history, boolean randomize) throws IOException{
        super.start(history, randomize);
        List<PlayerStore> players = getPlayersSortedByAccountOrKicked();
        PlayerStore p1 = players.get(0);
        PlayerStore p2 = players.get(1);
        assignPoints(p1, p2);
    }

    /**
     * 
     * Given two player stores, assign points based on their kicked state and
     * accounts.
     * 
     * Method is made public for testing purposes
     * 
     */
    public void assignPoints(PlayerStore p1, PlayerStore p2){
        // check for ties
        if (!(p1.wasKicked() && p2.wasKicked())
                && (p1.wasKicked() || p2.wasKicked() || Util.lessThan(p2.account, p1.account))) {
            // p1 wins, p2 looses
            result.setPlayerPoints(p1.getSpec().getName(), WIN_POINTS);
            result.setPlayerPoints(p2.getSpec().getName(), LOSS_POINTS);
        } else {
            // Tie
            result.setPlayerPoints(p1.getSpec().getName(), TIE_POINTS);
            result.setPlayerPoints(p2.getSpec().getName(), TIE_POINTS);
        }
    }

    public Result getResult(){
        return result;
    }

    static public class Result {

        private final String player1;
        private final String player2;
        private int player1Points;
        private int player2Points;

        public Result(String player1, String player2) {
            this.player1 = player1;
            this.player2 = player2;
        }

        public void setPlayerPoints(String playerName, int points){
            if (playerName.equals(player1)) {
                this.player1Points = points;
            } else if (playerName.equals(player2)) {
                this.player2Points = points;
            } else {
                unparticipatingPlayer(playerName);
            }
        }

        private int unparticipatingPlayer(String playerName){
            throw new RuntimeException("Player: " + playerName + " didn't participate in game!");
        }

        public void setPlayer2Points(int player2Points){
            this.player2Points = player2Points;
        }

        public int getPoints(String playerName){
            if (playerName.equals(player1)) {
                return player1Points;
            }
            if (playerName.equals(player2)) {
                return player2Points;
            }
            return unparticipatingPlayer(playerName);
        }

        @Override
        public String toString(){
            return player1 + " earns " + player1Points + " Points\n" + player2 + " earns " + player2Points
                    + " Points\n";

        }

    }
}
