/* Util.java
 * Bryan Chadwick :: 2007
 * Utility methods/functions */

package edu.neu.ccs.demeterf.util;

import edu.neu.ccs.demeterf.dispatch.MethodDB;
import edu.neu.ccs.demeterf.dispatch.DBEntry;
import edu.neu.ccs.demeterf.lib.List;
import edu.neu.ccs.demeterf.lib.RE;
import edu.neu.ccs.demeterf.lib.ident;
import edu.neu.ccs.demeterf.lib.verbatim;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Constructor;
import java.lang.reflect.Modifier;
import java.util.HashSet;
import java.util.HashMap;

/** Utility Class for globally useful methods */
public class Util {
    private Util(){}
    
    /** Show Methods for Debugging */
    static boolean debugOn = false;
    /** Show Method selection Warnings */
    public static boolean warningOn = false;
    /** Allow null fields during traversal */
    public static boolean allowNull = false;
    /** Skip private fields during Traversal, this is set to true to be
     *    able to hide the library details, so we can use non Comparable
     *    types with an external Comparator */
    public static boolean skipPrivate = true;
    /** Turn Traversal debugging output on/off */
    public static boolean setDebug(boolean b){ return debugOn = b; }
    /** Turn Traversal warning output on/off */
    public static boolean setWarning(boolean b){ return warningOn = b; }
    /** Turn support for null fields on/off */
    public static boolean setAllowNull(boolean b){ return allowNull = b; }
    /** Skip Private Fields */
    public static boolean setSkipPrivate(boolean b){ return skipPrivate = b; }
    
    /** Shorter than System.out... */
    public static void print(String s){ if(debugOn)System.out.print(s); }
    /** Shorter than System.out... */
    public static void println(String s){ print(s+"\n"); }
    
    /** List of Primitive Classes */
    public static HashSet<Class<?>> builtIns = new HashSet<Class<?>>();
    public static Class<?> builtInArray[] = 
        { short.class,Short.class, Integer.class, int.class,
          Float.class, float.class, Long.class, long.class,Double.class, double.class,
          String.class, Boolean.class, boolean.class,char.class, Character.class,
          ident.class,verbatim.class};
    static{
        for(Class<?> c:builtInArray)builtIns.add(c);
    }
    /** Add a Class to be considered a Primitive */
    public static void addBuiltIn(Class<?> c){ builtIns.add(c); }
    /** Add a Number of Classes to be considered Primitives */
    public static void addBuiltIns(Class<?> ... cs){ for(Class<?> c:cs)builtIns.add(c); }
    
    private static java.util.HashMap<Class<?>, List<Field>>
        memo = new java.util.HashMap<Class<?>, List<Field>>();
    
    /** Get all the Fields of a given Class (transitive) */
    public static List<Field> getFuncFields(Class<?> c){
        if(memo.containsKey(c))return memo.get(c);
        synchronized(memo){
            Class<?> cc = c;
            List<Field> lst = List.<Field>create();
            while(c != null && !c.isPrimitive()){
                for(Field f:c.getDeclaredFields()){
                    int m = f.getModifiers();
                    if(!(Modifier.isStatic(m) || (skipPrivate && Modifier.isPrivate(m))))
                        lst = lst.push(f);
                }   
                c = c.getSuperclass();
            }
            lst = lst.reverse();
            memo.put(cc, lst);
            return lst;
        }
    }
    
    /** Add the Argument (if there is one) to the end of the array */
    public static Object[] addArg(Object o[], Option arg){
        Object r[] = o;
        if(arg.some()){
            r = new Object[o.length+1];
            int i = 0;
            for(; i < o.length; i++)r[i] = o[i];
            r[i] = arg.get();
        }
        return r;
    }
    
    /** Generic function object application (Faster) */
    public static Object applyFObj(Object f, Object o[], MethodDB<Method> db, String meth, int def){
        Class<?> cs[] = classesFromObjects(o, o.length);
        DBEntry<Method> ml = db.matchEntryFast(cs);
        
        if(ml == null){
            if(def < 0)
                throw new RuntimeException("\n  DemeterF: Did Not Find a Match for: \n      "+
                        signature(f.getClass(), meth, o, o.length)+"\n");
            return o[def];
        }
        return applyMethod(ml.getMethod(),f,o);
    }
    /** Apply the given Method to a subsequence of the given arguments */
    public static <R> R applyMethod(Method m, Object targ, Object[] args){
        try{
            if(!m.isAccessible())m.setAccessible(true);
            R ret = (R)m.invoke(targ, objectSubset(args, m.getParameterTypes().length));
            return ret;
        }catch(java.lang.IllegalAccessException iae){
            throw new RuntimeException(iae);
        }catch(java.lang.reflect.InvocationTargetException ite){
            if((ite.getTargetException() instanceof RuntimeException))
                throw (RuntimeException)ite.getCause();
            if((ite.getTargetException() instanceof Error))
                throw (Error)ite.getCause();
            //ite.getTargetException().printStackTrace();
            throw new RuntimeException(ite.getCause());
        }
    }
    /** Output warnings if there are methods with more specific argument types
     *    that are not selected because they have less arguments. */
    static void checkWarnMethods(List<DBEntry<Method>> lst, Class<?> c){
        if(!warningOn)return;
        //System.err.println(" Warning Check");
        DBEntry<Method> theOne = lst.top();
        while(!(lst = lst.pop()).isEmpty()){
            DBEntry<Method> e = lst.top();
            if(e.numArgs() < theOne.numArgs() && e.numArgs() > 0)
                if(theOne.arg(0).isAssignableFrom(e.arg(0)) &&
                   !theOne.arg(0).equals(e.arg(0))){
                    System.err.println(
                            "\n %% DemeterF Warning: A method with more arguments has been\n"+
                            "      chosen instead of one with a more specific first argument\n"+
                            "  function class: "+c.getSimpleName()+"\n"+
                            "  * Chose : "+dbEntrySig(theOne)+"\n"+
                            "  *   Not : "+dbEntrySig(e)+"\n");
                    return;
                }
        }
    }
    
    /** Convert a DBEntry to a String */
    private static String dbEntrySig(DBEntry<?> e){ return e.toString(); }
    
    /** Class Array from an Object Array [Map :)] */
    public static Class<?>[] classesFromObjects(Object o[], int len){
        Class<?> cs[] = new Class[len];
        for(int i = 0; i < len; i++)
            if(o[i]==null){
                if(Util.allowNull)cs[i] = null;
                else nullError();
            }else
                cs[i] = o[i].getClass();
        return cs;
    }
    /** Object Array from another Object Array */
    public static Object[] objectSubset(Object o[], int len){
        if(o.length == len)return o;
        Object os[] = new Object[len];
        for(int i = 0; i < len; i++)
            os[i] = o[i];
        return os;
    }
    /** Signature Printing Helper */
    public static String signature(Method m){
        return signature(m.getDeclaringClass(), m.getName(),
                m.getParameterTypes(), m.getParameterTypes().length);
    }
    /** Signature Printing Helper */
    public static String signature(Constructor<?> c){
        return signature(c.getDeclaringClass(), c.getDeclaringClass().getSimpleName(),
                c.getParameterTypes(), c.getParameterTypes().length);
    }
    /** Signature Printing Helper */
    public static String signature(Class<?> c, String name, Object o[], int max){
        return signature(c, name, classesFromObjects(o, max), max);
    }
    /** Signature Printing Helper */
    public static String signature(Class<?> c, String name, Class<?> cs[], int max){
        String[] ss = new String[cs.length];
        for(int i = 0; i < max; i++)ss[i] = cs[i].getSimpleName();
        return signature(c, name, ss, max);
    }
    /** Signature Printing Helper */
    public static String signature(Class<?> c, String name, String cs[], int max){
        String s = c.getSimpleName()+"."+name+"(";
        for(int i = 0; i < max; i++){
            s += (cs == null)?"null":cs[i];
            if(i < max-1)s += (", ");
        }
        return s+")";
    }
    /** Throw a Traversal Error for a Null Object */
    public static void nullError(){
        throw new RuntimeException("\n  DemeterF: Null Object Found\n");
    }
    /** Throw a Traversal Error for a Null Field */
    public static void nullFieldError(Field f){
        throw new RuntimeException("\n  DemeterF: Null Field Found:  "+
                f.getDeclaringClass().getSimpleName()+"."+f.getName()+"\n");
    }
    
    /** For predicting Box/Unboxing and the associated Type differences */
    public static Class<?> box(Class<?> c){
        if(c.isPrimitive())return unboxed.get(c);
        return c;
    }
    static HashMap<Class<?>, Class<?>> unboxed = new HashMap<Class<?>, Class<?>>();
    static{
        unboxed.put(short.class, Short.class);
        unboxed.put(int.class, Integer.class);
        unboxed.put(long.class, Long.class);
        unboxed.put(float.class, Float.class);
        unboxed.put(double.class, Double.class);
        unboxed.put(char.class, Character.class);
        unboxed.put(byte.class, Byte.class);
        unboxed.put(boolean.class, Boolean.class);
    }
    
    /** For predicting Box/Unboxing and the associated Type differences */
    public static Class<?> unbox(Class<?> c){
        if(boxed.containsKey(c))return boxed.get(c);
        return c;
    }
    static HashMap<Class<?>, Class<?>> boxed = new HashMap<Class<?>, Class<?>>();
    static{
        boxed.put(Short.class, short.class);
        boxed.put(Integer.class, int.class);
        boxed.put(Long.class, long.class);
        boxed.put(Float.class, float.class);
        boxed.put(Double.class, double.class);
        boxed.put(Character.class, char.class);
        boxed.put(Byte.class, byte.class);
        boxed.put(Boolean.class, boolean.class);
    }
    
    /** Return the given string with a capitalized first character */
    public static String capCase(String s){
        if(s.length() == 0)return s;
        return Character.toUpperCase(s.charAt(0))+s.substring(1);
    }
    /** Write a file... */
    public static String writeFile(String name, String suf, String cls, String dir, String h){
        String file = dir+java.io.File.separatorChar+name+suf;
        try{
            PrintStream out = new PrintStream(new FileOutputStream(file));
            out.println(h);
            out.println(cls);
            out.close();
        }catch(FileNotFoundException fnf){
            throw new RE("Unable to Write File: "+file+"\n"+
                    "   * Please Check that directories exist and\n"+
                    "       permissions are set correctly");
        }
        return cls;
    }    
}