package edu.neu.ccs.demeterf.lib;

import java.util.Comparator;

import edu.neu.ccs.demeterf.Fields;

/** Represents a Node of an RBTree, e.g., the non-empty tree */
public class RBNode<X extends Comparable<X>> extends RBTree<X>{
    protected final RBColor color;
    protected final X data;
    protected final RBTree<X> left;
    protected final RBTree<X> right;
    private final int hash;
    
    public RBNode(RBColor color, X data, RBTree<X> left, RBTree<X> right){
        this.color = color;
        this.data = data;
        this.left = left;
        this.right = right;
        hash = data.hashCode()+left.hashCode()+right.hashCode();
    }
    public String toString(){ return "(node "+color+" "+data+" "+left+" "+right+")"; }
    public boolean equals(Object o){
        if(!(o instanceof RBNode))return false;
        RBNode oo = (RBNode)o;
        return (((Object)color).equals(oo.color))&&
        (((Object)data).equals(oo.data))&&
        (((Object)left).equals(oo.left))&&
        (((Object)right).equals(oo.right));
    }

    public static class color extends Fields.any{}
    public static class data extends Fields.any{}
    public static class left extends Fields.any{}
    public static class right extends Fields.any{}

    public boolean isLeaf(){ return false; }
    boolean isBlack(){ return color.isBlack(); }
    boolean isRed(){ return color.isRed(); }
    RBNode<X> asNode(){ return this; }

    public boolean contains(X x, Comparator<X> comp){
        int c = comp.compare(x, data);
        if(c < 0)return left.contains(x, comp);
        if(c > 0)return right.contains(x, comp);
        return true;
    }
    public boolean containsAll(RBTree<X> x, Comparator<X> comp){
        if(x.isLeaf())return true;
        RBNode<X> n = x.asNode();
        return (contains(n.data,comp) &&
                containsAll(n.left,comp) &&
                containsAll(n.right,comp));
    }
    public X find(X x, Comparator<X> comp){
        int c = comp.compare(x, data);
        if(c < 0)return left.find(x, comp);
        if(c > 0)return right.find(x, comp);
        return data;
    }

    public RBTree<X> remove(X x, Comparator<X> c){ return del(x,c).makeBlack(); }

    RBTree<X> del(X x, Comparator<X> comp){
        int c = comp.compare(x,data);
        if(c < 0){
            return left.isBlackNode()?
                    balleft(left.del(x,comp),data,right):
                        node(red(),data,left.del(x,comp),right);
        }
        if(c > 0){
            return right.isBlackNode()?
                    balright(left,data,right.del(x,comp)):
                        node(red(),data,left,right.del(x,comp)); 
        }
        return append(left,right);
    }
    // This should be done better (faster) by passing arguments ("continuation")
    public List<X> toList(){ return left.toList().append(right.toList().push(data)); }
    public int size(){ return 1+left.size()+right.size(); }
    
    RBTree<X> makeBlack(){ return node(black(),data,left,right); }
    RBTree<X> makeRed(){ return node(red(),data,left,right); }
    public X pred(){ return left.max(); }
    public X succ(){ return right.min(); }
    public X min(){ return left.isLeaf()?this.data:left.min(); }
    public X max(){ return right.isLeaf()?this.data:right.min(); }

    RBTree<X> ins(X x, Comparator<X> comp){
        int c = comp.compare(x, data);
        if(c == 0)return this;
        if(color.isBlack()){
            if(c < 0)return balance(data,left.ins(x, comp),right);
            return balance(data,left,right.ins(x, comp));
        }else{
            if(c < 0)return node(red(),data,left.ins(x, comp),right);
            return node(red(),data,left,right.ins(x, comp));
        }
    }
    public RBTree<X> replace(X x, Comparator<X> comp){
        int c = comp.compare(x, data);
        if(c == 0)return node(color,x,left,right);
        if(c < 0)return node(color,data,left.replace(x, comp),right);
        return node(color,data,left,right.replace(x, comp));
    }
    
    public int hashCode(){ return hash; }
}

