/* EdgeRemove.java
 * Bryan Chadwick :: 2007
 * Adds the ability to remove/ignore edges and modify BuiltIns
 *   while Traversing */

package edu.neu.ccs.demeterf.control;

import edu.neu.ccs.demeterf.dispatch.Type;
import edu.neu.ccs.demeterf.lib.List;

import java.lang.reflect.Field;

/** Supports the ability to ignore/remove edges of a structure when Traversing.  Essentially
 *   it is a boolean function object: 
 *   <blockquote>
 *      <tt>boolean ignore(Class, String);</tt>
 *   </blockquote>
 *   that allows Per-Traversal noise elimination.  See {@link edu.neu.ccs.demeterf.Traversal Traversal}
 *   for how its used in construction, and {@link edu.neu.ccs.demeterf.examples Examples}
 *   for how to eliminate uneeded fields during traversal.
 */
public class EdgeRemove extends EdgeBypass{
    protected List<Edge> ignorelist = List.create();

    public EdgeRemove(Edge ... edges){ ignorelist = List.create(edges); }
    /** Remove all the edges of the given classes */
    public EdgeRemove(Class<?> ... cs){ for(Class<?> c:cs)this.addRemove(c); }
    
    /** Takes a String in the format of "Class.edge "+ (meaning more than one).  We then attempt
      *   to recreate an EdgeRemove with the given edges removeed. Static methods within
      *   {@link edu.neu.ccs.demeterf.dispatch.Type} can be used to extend the path that is
      *   used to search for classes. */
    public EdgeRemove(String ... remove){ ignorelist = init("", remove); }
    public EdgeRemove(String remove){ ignorelist = init("", remove); }
    public EdgeRemove(String pkg, String remove){ ignorelist = init(pkg+".", remove); }
    
    public void addRemove(Class<?> c, String f){ addRemove(new Edge(c,f)); }
    public void addRemove(Edge e){ ignorelist = ignorelist.push(e); }
    public void addRemove(Class<?> c){ 
        Field fs[] = c.getDeclaredFields();
        for(Field f:fs)
            addRemove(new Edge(c,f.getName()));
    }

    public boolean ignore(Class<?> c, String f){ return ignore(new Edge(c,f)); }
    public boolean ignore(Edge e){ return ignorelist.contains(e); }
}