package player;

import java.io.IOException;
import logging.Logger;
import edu.neu.ccs.demeterf.lib.List;
import edu.neu.ccs.demeterf.util.CLI;
import scg.Constants;

/** Main Player Class, Registers a player, and sets up a server */
public class PlayerMain {

    /** Options */
    static final String HELP = "--help", NOREG = "--noreg";
    static final List<String> allArgs = PlayerFactory.allOptions.append(List.create(HELP,NOREG));

    /** Print usage information and Quit */
    static void usage(boolean full, String err){
        if (err.length() > 0)
            System.err.println("  !! " + err + "\n");
        System.err.println("  ** Usage: PlayerMain <PlayerPort> <ServerAddr> " + "<TeamName> <Password>");
        if(full)
            p("\n    Options can be:\n"
                    + "      --help      : Print this message\n"
                    + "      --noreg     : Skip registration, assume the Admin knows where you\n"
                    + "                      are.  If you use noreg then you only need to provide\n"
                    + "                      your PlayerPort #.\n\n"
                    + "    * NOTE: For a real competition you cannot use 'noreg' and *must* give\n"
                    + "            all arguments.\n\n"
                    + "    Port : the socket port the Player listens on. Should be an integer > 1024\n"
                    + "    ServerAddr : Given by the Course Staff. This may be a host name/URL or\n"
                    + "                   an IP Address.\n"
                    + "    TeamName : must be the same as when you pre-registered.\n"
                    + "    Password : same as when you pre-registered.");
        // ******************************
        // PlayerFactory has other options we should document...
        else
            p("  ** Use --help for a full description");
        p("\n  ** Typical Usages will be:\n"
                + "      java player.PlayerMain 7000 aurail.ccs.neu.edu \"MyTeam\" \"MyPass\"\n"
                + "      java player.PlayerMain 7000 --noreg\n");
        System.exit(err.length() > 0 ? 1 : 0);
    }

    /**  */
    public static void p(String s){ System.err.println(s); }
    
    /** Run the Player... */
    public static void main(String[] argArr) throws Exception{
        List<String> split[] = CLI.splitArgs(argArr),
                     options = split[CLI.OPTS],
                     args = split[CLI.ARGS];
        p("\n               /-------------------------------------\\");
        p("               |           The BasicPlayer           |");
        p("               |             Version "+Player.REV+"             |");
        p("               |            Game Rev "+Constants.REV+"             |");
        p("               \\-------------------------------------/\n");
        
        if (options.contains(HELP))usage(true, "");
        List<String> unknown = CLI.invalidOptions(options, allArgs);
        if(!unknown.isEmpty())
            usage(false, "Unknown Option(s): "+unknown.toString(", ",""));
        
        // Is registration needed?
        boolean doreg = !options.contains(NOREG);
        if (args.length() < 1 || (doreg && args.length() != 4)) {
            usage(false, "Not enough manditory arguments");
            return;
        }
        // Player's Port Number is first
        int port = Integer.parseInt(args.lookup(0));

        // Other arguments are just for registration
        String serve = doreg ? args.lookup(1) : "", team = doreg ? args.lookup(2) : "No Name", pass = doreg ? args
                .lookup(3) : "";

        // Create a new Logger
        Logger log = Logger.text(System.out, scg.Util.logFileName("player"));

        // Register if Needed
        if (doreg) {
            if (!new Register(serve, log).doReg(team, pass, port))
                System.exit(1);
            log.event("Player '" + team + "' Registered Successfully");
        }
        // Start the Player Server... it will wait for its turn
        log.notify("Player Started for Team '" + team + "'");
        try {
            PlayerFactory.setGlobalOptions(options);
            PlayerServer.run(PlayerFactory.create(options), port, log);
        } catch (IOException ie) {
            log.error("IOException: " + ie.getMessage());
        }
        log.notify("Player Shutdown");
    }
}
