package player;

import edu.neu.ccs.demeterf.lib.*;
import edu.neu.ccs.demeterf.util.CLI;
import scg.gen.PlayerContext;
import logging.Logger;
import scg.gen.*;
import scg.Util;
import player.tasks.*;

public class PlayerFactory {
    /** The available command-line options */
    public static final String INFINITE = "--infiniteloop", NOOFFERS = "--nooffers", EMPTYTRANS = "--emptytrans",
                        EXCEPTION = "--exception", GIBERISH = "--giberish", SMART = "--smart",
                        BADPRICES = "--badoffers", NOSECRETS = "--nosecrets", ALLSECRETS = "--allsecrets", 
                        BADTYPES = "--badtypes", OFFERACCEPT = "--offeraccept", GOODSECRETS = "--goodsecrets",
                        NEWACCEPT = "--newaccept", OFFER = "--offer", PARSOLVE = "--parsolve", DUPTYPES = "--duptypes",
                        FASTPITCH = "--fastpitch";

    /** As a list... for handy searching */
    static final List<String> allOptions = List.create(INFINITE, NOOFFERS,
            EMPTYTRANS, EXCEPTION, GIBERISH, SMART, BADPRICES, NOSECRETS,
            ALLSECRETS, BADTYPES, OFFERACCEPT, GOODSECRETS, NEWACCEPT,
            OFFER, PARSOLVE, DUPTYPES, FASTPITCH);

    /** Global command-line options, so others can see what flags are set */
    static List<String> globalOptions;
    public static void setGlobalOptions(List<String> opts){ globalOptions = opts; }
    public static boolean globalOptionSet(String s){ return globalOptions.contains(s); }
    public static String opitonArgument(String s){ return CLI.getArgument(s, globalOptions); }
    
    /** Create a PlayerFactory with the given command line options */
    public static PlayerFactory create(List<String> opts){
        setGlobalOptions(opts);
        return new PlayerFactory();
    }

    /** Create a Player with teh given command line Options, context, and logger */
    public Player getAPlayer(PlayerContext ctxt, Logger l){
        if (globalOptionSet(SMART))return player.smarttasks.Player.create(ctxt, l);
        if (globalOptionSet(INFINITE))return new InfiniteLoop(ctxt, l);
        if (globalOptionSet(NOOFFERS))return new NoOffers(ctxt, l);
        if (globalOptionSet(EMPTYTRANS))return new EmptyTrans(ctxt, l);
        if (globalOptionSet(EXCEPTION))return new Except(ctxt, l);
        if (globalOptionSet(GIBERISH))return new Giberish(ctxt, l);
        if (globalOptionSet(BADPRICES))return new BadPrices(ctxt, l);
        if (globalOptionSet(BADTYPES))return new BadTypes(ctxt, l);
        if (globalOptionSet(OFFERACCEPT))return new OfferAccept(ctxt, l);
        if (globalOptionSet(DUPTYPES))return new DupTypes(ctxt, l);
        return Player.create(ctxt, l);
    }

    /* Different, Unruly Players... */

    /** Bad Players start here... */
    static class BadPlayer extends Player {
        BadPlayer(PlayerContext ctxt, Logger l) { super(ctxt, l); }
        public PlayerTrans playOne(){ return super.play(); }
        /** Be Bad about half the time */
        public PlayerTrans play(){
            if (Util.coinFlip())
                return super.play();
            return playOne();
        }
    }
    
    /** Infinite loop */
    static class InfiniteLoop extends BadPlayer {
        InfiniteLoop(PlayerContext ctxt, Logger l) { super(ctxt, l); }
        public PlayerTrans playOne(){
            while(true)
                try{
                    log.notify("Still Waiting");
                    Thread.sleep(10000);
                }catch(InterruptedException e){}
        }
    }

    /** Empty Transaction */
    static class EmptyTrans extends BadPlayer {
        EmptyTrans(PlayerContext ctxt, Logger l) { super(ctxt, l); }
        public PlayerTrans playOne(){
            return new PlayerTrans(context.getId(), List.<Transaction> create());
        }
    }

    /** No Offered Transactions */
    static class NoOffers extends BadPlayer {
        NoOffers(PlayerContext ctxt, Logger l){ super(ctxt, l); }
        public PlayerTrans playOne(){
            return new PlayerTrans(context.getId(), super.play()
                    .getTs().filterout(new List.Pred<Transaction>() {
                public boolean huh(Transaction t){ return t instanceof OfferTrans; }
            }));
        }
    }

    /** Bad Prices in Offers */
    static class BadPrices extends BadPlayer {
        BadPrices(PlayerContext ctxt, Logger l) { super(ctxt, l); }
        public OfferTrans change(OfferTrans t){
            double price = Math.random()*4-2;
            return new OfferTrans(ChallengeKind.ALL, t.getPred(), price);
        }
        
        public PlayerTrans playOne(){
            return new PlayerTrans(context.getId(), super.play().getTs()
                    .map(new List.Map<Transaction,Transaction>() {
                        public Transaction map(Transaction t){
                            if(!(t instanceof OfferTrans))
                                return t;
                            return change((OfferTrans)t);
                        }
                }));
        }
    }

    /** Bad Prices in Offers */
    static class DupTypes extends BadPlayer {
        DupTypes(PlayerContext ctxt, Logger l) { super(ctxt, l); }
        public PlayerTrans playOne(){
            return new PlayerTrans(context.getId(), super.play().getTs()
                    .filter(new List.Pred<Transaction>(){
                        public boolean huh(Transaction t){
                            return (!(t instanceof OfferTrans));
                        }
                }).append(List.<Transaction>create(
                        new OfferTrans(ChallengeKind.SECRET, new ProblemType(List.<Integer>create(7, 15)), 0.87),
                        new OfferTrans(ChallengeKind.SECRET, new ProblemType(List.<Integer>create(15, 7)), 0.87))
                        ));
        }
    }
    
    /** Bad Types in Offers */
    static class BadTypes extends BadPrices {
        BadTypes(PlayerContext ctxt, Logger l) { super(ctxt, l); }
        public OfferTrans change(OfferTrans t){
            return new OfferTrans(t.getKind(), 
                    new ProblemType(List.<Integer>create(Util.random(516)-258)), t.getPrice());
        }
    }
    
    /** Throws Exception Immediately */
    static class Except extends BadPlayer {
        Except(PlayerContext ctxt, Logger l){ super(ctxt, l); }
        public PlayerTrans playOne(){
            throw new RuntimeException("Random Error!!");
        }
    }

    /** Returns a nonsense response */
    static class Giberish extends BadPlayer {
        Giberish(PlayerContext ctxt, Logger l){ super(ctxt, l); }
        public String playString(){
            return "\"The man walked into the barn\"";
        }
           
    }
    
    /** Try to reoffer and accept the same challenge */
    static class OfferAccept extends BadPlayer {
        OfferAccept(PlayerContext ctxt, Logger l){ super(ctxt, l); }
        int acc;
        public PlayerTrans playOne(){
            final ReofferTask roff = this.reofferer(log);
            acc = -1;
            List<Transaction> offrs = (List)this.offerer(log).offer(context); 
            return new PlayerTrans(context.getId(), offrs.append(context.getTheirOffered()
                    .map(new List.Map<OfferedChallenge,Transaction>(){
                        public Transaction map(OfferedChallenge c){
                            if(acc < 0)
                                acc = c.getKey();
                            return roff.reoffer(c, context.getConfig());
                        }
                    })).append((Transaction)new AcceptTrans(acc)));
        }
           
    }
}
