package player;

import logging.Logger;
import player.tasks.AcceptTask;
import player.tasks.OfferTask;
import player.tasks.ProvideTask;
import player.tasks.ReofferTask;
import player.tasks.SolveTask;
import scg.gen.AcceptedChallenge;
import scg.gen.PlayerContext;
import scg.gen.PlayerID;
import scg.gen.PlayerTrans;
import scg.gen.Transaction;
import edu.neu.ccs.demeterf.lib.List;

/**
 * Player class, responsible for taking a single turn. We deligate to separate
 * Tasks classes to make things a little easier to understand. See
 * {@link player.tasks Tasks}.
 */
public class Player {
    /** Current Version */
    static final String REV = "$Rev: 504 $".substring(6,9);
    
    /** Team Name: change this to your team's name */
    private final String teamName = "BasicPlayer";
    /** The context for this current turn */
    protected final PlayerContext context;
    /** Logger instance for output */
    protected final Logger log;
    /** When were we created? */
    protected final long bigbang = System.currentTimeMillis(); 

    /** Get our assigned ID from the context */
    protected PlayerID playerID(){
        return context.getId();
    }

    protected Player(PlayerContext ctxt, Logger l) {
        context = ctxt;
        log = l;
    }

    /** Construct a new player with the given Context */
    public static Player create(PlayerContext ctxt, Logger l){
        l.notify("This is the *PLAIN* Player!!");
        return new Player(ctxt, l);
    }

    /** Get this Players Team Name */
    public String getName(){ return teamName; }

    /** Take a turn for this player... return a String */
    public String playString(){ return play().toString(); }
    /** Take a turn for this player */
    public PlayerTrans play(){
        log.event("Chance to Play...");
        // Try to accept some of the offered challenges
        List<Transaction> accs = acceptChallenges();
        // If not then we must Reoffer
        if (accs.isEmpty()) {
            accs = reofferChallenges();
        }
        // Create a player transaction from our ID and our other transactions
        return new PlayerTrans(playerID(), offerChallenges().append(
                accs.append(provideChallenges().append(solveChallenges()))));
    }

    /** Offer new Challenges... that are not currently offered */
    private List<Transaction> offerChallenges(){
        log.event("Running Offer");
        return (List) offerer(log).offer(context);
    }

    /** Accept Challenges that seem reasonable/profitable */
    private List<Transaction> acceptChallenges(){
        log.event("Running Accept");
        return (List) accepter(log).accept(context);
    }

    /** Reoffer other Player's Challenges */
    private List<Transaction> reofferChallenges(){
        log.event("Running Reoffer");
        final ReofferTask reoff = new ReofferTask(log);
        return (List) reoff.reoffer(context.getTheirOffered(), context.getConfig());
    }

    /** Provide Problem instances for accepted Challenges */
    private List<Transaction> provideChallenges(){
        log.event("Running Provide");
        final ProvideTask prov = provider(log);
        return context.getAccepted().map(new List.Map<AcceptedChallenge, Transaction>(){
            public Transaction map(AcceptedChallenge ch){
                return prov.provide(ch);
            }
        });
    }

    /** Solve provided Problem instances */
    private List<Transaction> solveChallenges(){
        log.event("Running Solve");
        final SolveTask solve = solver(log);
        return solve.solve(context.getProvided(),
                context.getConfig().getProfitFactor(), bigbang);
    }
    
    protected OfferTask offerer(Logger log){ return new OfferTask(log, context.getConfig()); }
    protected AcceptTask accepter(Logger log){ return new AcceptTask(log); }
    protected ProvideTask provider(Logger log){ return new ProvideTask(log); }
    protected SolveTask solver(Logger log){ return new SolveTask(log); }
    protected ReofferTask reofferer(Logger log){ return new ReofferTask(log); }
}
