package rules;

import junit.framework.Assert;

import org.junit.Before;
import org.junit.Test;

import scg.Claim;
import scg.ParseException;
import scg.net.PlayerSpec;
import scg.net.admin.RemotePlayerProxy;
import scg.protocolInterpreter.RefutationProtocolInterpreter;
import csp.CSPInstance;
import csp.CSPSolution;


public class TestRefutationProtocolInterpreter {
	
	private static final int INITIAL_REPUTATION = 100;
	private static final int TIMEOUT = 60;
	private static final double DELTA = 0.8;
	private RemotePlayerProxy alice;
	private RemotePlayerProxy bob;
	
	@Before
	public void initTest() {
		alice = new RemotePlayerProxy(new PlayerSpec("alice", "localhost", -1), TIMEOUT, INITIAL_REPUTATION);
		bob = new RemotePlayerProxy(new PlayerSpec("bob", "localhost", -1), TIMEOUT, INITIAL_REPUTATION);
	}

	@Test
	public void testCorrectNextStepInPositiveSecret() throws ParseException, csp.ParseException {
		Claim claim = Claim.parse("csp.CSPInstanceSet {{ (12 22) }} scg.protocol.PositiveSecret {{ }} 0.444444 0.8");
		RefutationProtocolInterpreter rpi = new RefutationProtocolInterpreter(alice, bob, claim);
		
		Assert.assertFalse(rpi.isComplete());
		
		rpi.setInstance(CSPInstance.parse("x1 x2 x3 (12 {1} x1 x2 x3)"));
		rpi.proceed();
		Assert.assertFalse(rpi.isComplete());
		
		rpi.setSolution(CSPSolution.parse("[(x1->true) (x2->true) (x3->false)]"));
		rpi.proceed();
		
		rpi.setSolution(CSPSolution.parse("[(x1->true) (x2->true) (x3->true)]"));
		rpi.proceed();
		rpi.proceed();
		Assert.assertEquals(-1d, rpi.getResult()); // Bob Wins
		Assert.assertEquals(INITIAL_REPUTATION + DELTA, bob.getReputation());
		Assert.assertEquals(INITIAL_REPUTATION - DELTA, alice.getReputation());
		
		Assert.assertTrue(rpi.isComplete());
	}

	@Test
	public void testCorrectNextStepInNegativeSecret() throws ParseException, csp.ParseException {
		Claim claim = Claim.parse("csp.CSPInstanceSet {{ (12 22) }} scg.protocol.NegativeSecret {{ }} 0.444444 0.8");
		RefutationProtocolInterpreter rpi = new RefutationProtocolInterpreter(alice, bob, claim);
		Assert.assertFalse(rpi.isComplete());
		
		rpi.setInstance(CSPInstance.parse("x1 x2 x3 (12 {1} x1 x2 x3)"));
		rpi.proceed();
		Assert.assertFalse(rpi.isComplete());
		
		rpi.setSolution(CSPSolution.parse("[(x1->true) (x2->true) (x3->false)]"));
		rpi.proceed();
		
		rpi.setSolution(CSPSolution.parse("[(x1->true) (x2->true) (x3->true)]"));
		rpi.proceed();
		rpi.proceed();
		Assert.assertEquals(1d, rpi.getResult()); // Alice Wins
		Assert.assertEquals(INITIAL_REPUTATION + DELTA, alice.getReputation());
		Assert.assertEquals(INITIAL_REPUTATION - DELTA, bob.getReputation());
	}

}

