package csu670.actions;

import java.io.*;
import java.util.*;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITypedRegion;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.TypedRegion;

import csu670.actions.exceptions.ClassDictionaryException;
import csu670.actions.exceptions.StrategyDeclarationException;
import csu670.actions.exceptions.ValidationException;
import csu670.actions.exceptions.XAspectDocumentException;
import csu670.actions.exceptions.XPathDeclarationException;
import csu670.editors.Config;
import csu670.editors.scanners.XAspectPartitionScanner;
import csu670.generated.*;
import edu.neu.ccs.demeter.Text;
import edu.neu.ccs.demeter.aplib.sg.Strategy;
import edu.neu.ccs.demeter.aplib.Traversal;
import edu.neu.ccs.demeter.aplib.Traversal.EdgeSet;
import edu.neu.ccs.demeter.aplib.cd.ParseException;
import edu.neu.ccs.demeter.aplib.cd.Token;
import edu.neu.ccs.demeter.aplib.EdgeI;
import edu.neu.ccs.demeter.dj.ClassGraph;




/**
 * Validates the syntax of the XAspect document.  This class 
 * only validates syntax, and does not do semantic checking.
 * 
 * @author Pok
 */
public class XAspectSyntaxValidator extends ValidatorBase {
	

	public XAspectSyntaxValidator() {
	}
	

	
	private void validatePartitions(IDocument doc) throws ValidationException {
		int cdPartitionCount = 0;
		int trPartitionCount = 0;
		ITypedRegion[] partitions = doc.getDocumentPartitioner().computePartitioning(0, doc.getLength());
		for (int i=0; i<partitions.length; i++) {
			if (partitions[i].getType().equals(XAspectPartitionScanner.XASPECT_CD)) {
				cdPartitionCount++;
			} else if (partitions[i].getType().equals(XAspectPartitionScanner.XASPECT_CD)) {
				trPartitionCount++;
			}
		}
		
		if (cdPartitionCount > 1) {
			throw new XAspectDocumentException("You cannot have more than 1 cd partition.");
		}
		
		if (trPartitionCount > 1) {
			throw new XAspectDocumentException("You cannot have more than 1 traversal partition.");
		}
	}
	
	private XAspectDocument validateXAspectDocument(IDocument doc) {
		try {
			// HACK: We want to catch parser exception so we can show 
			// the correct line number in the future, thats why we want 
			// it to call the parse(Reader) instead of parse(String) 
			// method
			return XAspectDocument.parse(new StringReader(doc.get()));
		} catch (Throwable ex) {
			throw new XAspectDocumentException(ex);
		}
		
	}
	
	
	private void validateClassDictionary(IDocument doc, XAspectDocument docObj) throws ValidationException {
		String cd = docObj.get_cdSection().get_body().get_text().toString();
		edu.neu.ccs.demeter.aplib.cd.ClassGraph cg;
		try {
			// HACK: We want to catch parser exception so we can show 
			// the correct line number in the future, thats why we want 
			// it to call the parse(Reader) instead of parse(String) 
			// method
			cg = ClassGraph.parse(new StringReader(cd));
		} catch (ParseException ex) {
			try {
				// Try to fix line number 
				
				// Find the partition
				ITypedRegion[] partitions = doc.getDocumentPartitioner().computePartitioning(0, doc.getLength());
				int partitionIndex = 0;
				for (int i=0; i<partitions.length; i++) {
					if (partitions[i].getType().equals(XAspectPartitionScanner.XASPECT_CD)) {
						partitionIndex = i;
						break;
					}
				}
				
				// Calculate line number offset, offset start at line 0, so add 1
				int lineOffset = doc.getLineOfOffset(partitions[partitionIndex].getOffset());
				int parserLineNumber = ex.currentToken.next.beginLine;
				int lineNumber = parserLineNumber + lineOffset;
				int col = ex.currentToken.next.beginColumn;
				
				// Create the message
				String line = getLine(doc, lineNumber-1);
				StringBuffer sb = new StringBuffer();
				sb.append(line);
				sb.append(doc.getLineDelimiter(lineNumber));
				
				col--;
				for (int j=0; j<col; j++) {
					char c = line.charAt(j);
					// HACK: In demeterj, a tab is worth 8 cols
					if (c == '\t') {
						sb.append('\t');
						col -= 7;
					} else {
						sb.append(c);
					}
				}
				

				sb.append('^');
				sb.append(doc.getLineDelimiter(lineNumber));
				sb.append(doc.getLineDelimiter(lineNumber));
				sb.append(rebuildExceptionString(ex, lineNumber));
				


				throw new ClassDictionaryException(sb.toString());

				
			} catch (BadLocationException ex2) {
				// can't fix line number, oh well
				throw new ClassDictionaryException(ex);
			}
			
			
		} catch (Throwable ex) {
			throw new ClassDictionaryException(ex);
		}
	}
	
	
	
	
	
	private void validateXPaths(XAspectDocument docObj) throws ValidationException {
		String[] contextNodes = listContextNode(docObj);
		String[] exps = this.listXPath(docObj);
		
		if (contextNodes.length >= 2) {
			throw new XPathDeclarationException("You cannot declare more than one context nodes.");
		} else if (contextNodes.length == 0 && exps.length > 0) {
			throw new XPathDeclarationException("You must declare a context node.");
		}
		
		for (int i=0; i<exps.length; i++) {
			// TODO: Validate syntax of xpath
			
		}
		
		
	}
	
	private void validateStrategies(XAspectDocument docObj) throws ValidationException {
		String[] strings = this.listStrategies(docObj);
		for (int i=0; i<strings.length; i++) {
			try {
				Strategy.fromString(strings[i]);
			} catch (Throwable ex) {
				throw new StrategyDeclarationException(ex);
			}
		}
	}
	
	private String getLine(IDocument doc, int lineNumber) throws BadLocationException {
		IRegion region = doc.getLineInformation(lineNumber);
		return doc.get(region.getOffset(), region.getLength());
	}
	
	/**
	 * Do the validation with the given contents.  
	 * @param doc the document to validate
	 * 
	 * @throws ValidationException
	 */
	public void validate(IDocument doc) throws ValidationException{
		// Validate the partition
		validatePartitions(doc);
			
		// Validate the syntax of the entire document.
		XAspectDocument docObj = validateXAspectDocument(doc);
		
		validateClassDictionary(doc, docObj);
		// Validate the sytax of the class dictionary.
		
		
		validateStrategies(docObj);
		

		validateXPaths(docObj);
	}
	
	
	
	
	
	
	/**
	 * The following code was copy from the constructor of ParseException
	 * @return
	 */
	private String rebuildExceptionString(ParseException ex, int newLineNumber) {
		int[][] expectedTokenSequences = ex.expectedTokenSequences;
		String[] tokenImage = ex.tokenImage;
		String eol = System.getProperty("line.separator", "\n");
		Token currentToken = ex.currentToken;
		
		String expected = "";
	    int maxSize = 0;
	    for (int i = 0; i < expectedTokenSequences.length; i++) {
	      if (maxSize < expectedTokenSequences[i].length) {
	        maxSize = expectedTokenSequences[i].length;
	      }
	      for (int j = 0; j < expectedTokenSequences[i].length; j++) {
	        expected += tokenImage[expectedTokenSequences[i][j]] + " ";
	      }
	      if (expectedTokenSequences[i][expectedTokenSequences[i].length - 1] != 0) {
	        expected += "...";
	      }
	      expected += eol + "    ";
	    }
	    String retval = "Encountered \"";
	    Token tok = currentToken.next;
	    for (int i = 0; i < maxSize; i++) {
	      if (i != 0) retval += " ";
	      if (tok.kind == 0) {
	        retval += tokenImage[0];
	        break;
	      }
	      retval += add_escapes(tok.image);
	      tok = tok.next; 
	    }
	    retval += "\" at line " + newLineNumber + ", column " + currentToken.next.beginColumn;
	    retval += "." + eol;
	    if (expectedTokenSequences.length == 1) {
	      retval += "Was expecting:" + eol + "    ";
	    } else {
	      retval += "Was expecting one of:" + eol + "    ";
	    }
	    retval += expected;
	    return retval;
	}
	
	/**
	 * The following code was copy from ParseException
	 */
	protected String add_escapes(String str) {
	      StringBuffer retval = new StringBuffer();
	      char ch;
	      for (int i = 0; i < str.length(); i++) {
	        switch (str.charAt(i))
	        {
	           case 0 :
	              continue;
	           case '\b':
	              retval.append("\\b");
	              continue;
	           case '\t':
	              retval.append("\\t");
	              continue;
	           case '\n':
	              retval.append("\\n");
	              continue;
	           case '\f':
	              retval.append("\\f");
	              continue;
	           case '\r':
	              retval.append("\\r");
	              continue;
	           case '\"':
	              retval.append("\\\"");
	              continue;
	           case '\'':
	              retval.append("\\\'");
	              continue;
	           case '\\':
	              retval.append("\\\\");
	              continue;
	           default:
	              if ((ch = str.charAt(i)) < 0x20 || ch > 0x7e) {
	                 String s = "0000" + Integer.toString(ch, 16);
	                 retval.append("\\u" + s.substring(s.length() - 4, s.length()));
	              } else {
	                 retval.append(ch);
	              }
	              continue;
	        }
	      }
	      return retval.toString();
	   }
	
	
}