package csu670.actions;

import java.io.IOException;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkbenchWindowActionDelegate;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.jface.dialogs.MessageDialog;

import csu670.actions.exceptions.ValidationException;
import csu670.editors.XAspectEditor;
import csu670.editors.scanners.XAspectPartitionScanner;


/**
 * Our sample action implements workbench action delegate.
 * The action proxy will be created by the workbench and
 * shown in the UI. When the user tries to use the action,
 * this delegate will be created and execution will be 
 * delegated to it.
 * @see IWorkbenchWindowActionDelegate
 */
public class ValidateAction implements IWorkbenchWindowActionDelegate {
	private IWorkbenchWindow window;
	/**
	 * The constructor.
	 */
	public ValidateAction() {
	}

	/**
	 * The action has been activated. The argument of the
	 * method represents the 'real' action sitting
	 * in the workbench UI.
	 * @see IWorkbenchWindowActionDelegate#run
	 */
	public void run(IAction action) {
		IWorkbenchPage activePage = window.getActivePage();
		if (activePage == null) return;
		
		IEditorPart activeEditor = activePage.getActiveEditor();
		if (activeEditor == null) return;
		
		if (activeEditor instanceof XAspectEditor) {
			XAspectEditor editor = (XAspectEditor)activeEditor;
			IDocumentProvider provider = editor.getDocumentProvider();
			IDocument doc = provider.getDocument(editor.getEditorInput());

			XAspectSyntaxValidator v = new XAspectSyntaxValidator();
			try {
				v.validate(doc);
				MessageDialog.openInformation(
						window.getShell(),
						"Validation Completed", 
						"Everything looks ok.");
			} catch (ValidationException e) {
				MessageDialog.openError(
						window.getShell(),
						e.getTitleText(), 
						e.getBodyText(doc));
			}
		} else {
			MessageDialog.openInformation(
				window.getShell(),
				"Csu670 Plug-in",
				"Invalid editor opened.");
		}
	}

	/**
	 * Selection in the workbench has been changed. We 
	 * can change the state of the 'real' action here
	 * if we want, but this can only happen after 
	 * the delegate has been created.
	 * @see IWorkbenchWindowActionDelegate#selectionChanged
	 */
	public void selectionChanged(IAction action, ISelection selection) {
	}

	/**
	 * We can use this method to dispose of any system
	 * resources we previously allocated.
	 * @see IWorkbenchWindowActionDelegate#dispose
	 */
	public void dispose() {
	}

	/**
	 * We will cache window object in order to
	 * be able to provide parent shell for the message dialog.
	 * @see IWorkbenchWindowActionDelegate#init
	 */
	public void init(IWorkbenchWindow window) {
		this.window = window;
	}
}