package com.ibm.lab.soln.resources.nature_builder;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.SequenceInputStream;

import com.ibm.lab.soln.resources.IResourceIDs;
/**
 * Processes an <code>IResourceDelta</code> to find .readme files and 
 * create stub .html files. Designed to be invoked as part of builder.
 * 
 * <p>
 * Source includes many <code>System.out.println()</code> trace entries that have
 * been commented out.  If you want to visualize the flow of build processing, 
 * uncomment these statements before starting a test cycle.
 * </p>
 * @see org.eclipse.core.resources.IResourceDelta
 * @see org.eclipse.core.resources.IResourceDeltaVisitor
 */
public class ReadmeVisitor implements IResourceDeltaVisitor {
  // switch to control write of trace data
  private boolean traceEnabled = false;
  // Value used to prefix trace messages
  String buildVisitMode = "\t\t\t\t\t>------><visit> ";

  /**
   * Required implementation of the visit method.  The <code>IResourceDelta</code> is 
   * processed by providing it a visitor using the <code>accept()</code> method
   * and using this method in the visitor to process any events of interest.
   * 
   * Processing continues as long as this method returns true or when the end 
   * of the <code>IResourceDelta</code> has been reached.
   */
  public boolean visit(IResourceDelta delta) throws CoreException {

    IResource resource = delta.getResource();
    int type = resource.getType();

    switch (delta.getKind()) {
      case IResourceDelta.ADDED :
        traceMsg("< Added: " + resource);
        if (type == IResource.FILE) {
          if (resource.getName().endsWith(".readme")) {
            processReadme(resource, resource.getParent());
          }
        }
        break;

      case IResourceDelta.CHANGED :
        traceMsg("< Changed: " + resource);
        if (type == IResource.FILE) {
          if (resource.getName().endsWith(".readme")) {
            processReadme(resource, resource.getParent());
          }
        }
        break;

      case IResourceDelta.REMOVED :
        traceMsg("< Removed: " + resource);
    }
    return true; // carry on
  }
  /**
   * Implements processing for a given .readme resource. 
   * This includes creation of a matching .html file stub with 
   * limited hard coded content. 
   * <p>
   * The .html file creation is forced. This means
   * the create will succeed even if a .html file with the same name exists on
   * the file system but not in the workspace.
   * </p>
   * @param org.eclipse.core.resources.IContainer
   * @param org.eclipse.core.resources.IResource
   */
  public void processReadme(IResource resource, IContainer container) {

    traceMsg("Found Readme Resource:");
    traceMsg("\t" + resource);
    InputStream stream = getReadmeContent((IFile) resource);

    try {
      String fileName = resource.getName();
      fileName = fileName.substring(0, (fileName.length() - 7));
      IFile htmlFile = container.getFile(new Path(fileName + ".html"));

      if (!htmlFile.exists()) {
        htmlFile.create(stream, true, null);
        htmlFile.setDerived(true);

      } else {
        htmlFile.setContents(stream, true, false, null);
        htmlFile.setDerived(true);
      }
      traceMsg("HTML file created.");

    } catch (CoreException e) {
      e.printStackTrace();
    }

  }

  /**
   * Wraps the <code>InputStream</code> available for the <code>IFile</code> 
   * resource with some HTML content.  Uses a simple approach, where a  
   * <code>ByteArrayInputStream</code> is used to convert a 
   * <code>StringBuffer</code> to a type of <code>InputStream</code> that can be 
   * joined with another using <code>SequenceInputStream</code>.
   * @param string
   * @return InputStream
   */
  public InputStream getReadmeContent(IFile readmeFile) {

    StringBuffer sbHeader = new StringBuffer();
    sbHeader.append(
      "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n");
    sbHeader.append("<HTML>\n");
    sbHeader.append("<HEAD>\n");
    sbHeader.append("<TITLE>Readme File as HTML</TITLE>\n");
    sbHeader.append("</HEAD>\n");
    sbHeader.append("<BODY>\n");
    sbHeader.append(
      "<H1>HTML Version of the " + readmeFile.getName() + "</H1>\n");
    sbHeader.append(
      "<P>The <CODE>ReadmeBuilder</CODE> has wrapped the .readme file content in a bit of html.</P>\n");
    sbHeader.append("<H3>Readme file data follows:</H3>\n");
    sbHeader.append("<HR><PRE>\n");

    ByteArrayInputStream header =
      new ByteArrayInputStream(sbHeader.toString().getBytes());

    StringBuffer sbFooter = new StringBuffer();
    sbFooter.append("</PRE><HR>\n");
    sbFooter.append(
      "<P>A simple approach. We'll leave the fancy formatting of the full content up to you.</P>\n");
    sbFooter.append("<P><STRONG>Enjoy...</STRONG></P>\n");
    sbFooter.append("</BODY>\n");
    sbFooter.append("</HTML>");

    ByteArrayInputStream footer =
      new ByteArrayInputStream(sbFooter.toString().getBytes());

    // Join the header and footer streams with the file stream
    SequenceInputStream top = null;
    InputStream total = null;
    try {
      InputStream file = readmeFile.getContents();
      top = new SequenceInputStream(header, file);
      total = (InputStream) new SequenceInputStream(top, footer);
    } catch (CoreException e) {
      total =
        (InputStream) new ByteArrayInputStream("Unable to get readme file contents.\n"
          .getBytes());

    }

    return total;
  }
  /**
    * Write trace statements.  
    * System.out.println with prefix tagging used for simplicity.
    */
  void traceMsg(String msg) {
    if (IResourceIDs.TRACE_ENABLED | traceEnabled)
      System.out.println(buildVisitMode + msg);
  }

}