package com.ibm.lab.soln.resources.nature_builder;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */
 
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.runtime.CoreException;

import com.ibm.lab.soln.resources.ActionResourceAction;
import com.ibm.lab.soln.resources.IResourceIDs;

/**
 * The <code>ActionRemoveCustomNature</code> implements an action 
 * to remove the nature implemented using <code>CustomNature</code> 
 * directly from an existing project. 
 * <p>
 * The <code>NewProjectWizard</code> creates a project with the 
 * <code>CustomNature</code>, this action removes it. If the nature 
 * was defined with a builder reference, the removal of the nature 
 * would also automatically remove the builder.
 * </p>
 * This demonstrates the API for the project and project description 
 * workspace resources and how you might implement this logic in your tool.
 * 
 * @see org.eclipse.core.resources.IProject
 * @see org.eclipse.core.resources.IProjectDescription
 * @see org.eclipse.core.resources.ICommand
 */
public class ActionRemoveCustomNature extends ActionResourceAction {

   /**
    * ActionRemoveCustomNature default constructor.
    */
   public ActionRemoveCustomNature() {
      super();
   }

   /**
    * Uses saved reference to selected project to call custom logic that
    * removes the nature from the current project. This logic depends on the 
    * nature being positioned as the first nature in the set known to the
    * project description.
    * 
    * @see org.eclipse.ui.IActionDelegate#run(IAction)
    * 
    * @param action action proxy that handles the presentation
    * portion of the plugin action
    */
   public void run(org.eclipse.jface.action.IAction action) {

      IProject project = (IProject) this.selection.iterator().next();

      try {
         this.removeNatureSpec(project);
         // close and open project to tickle out the nature image
         project.close(null);
         project.open(null);
         

      } catch (CoreException e) {
         e.printStackTrace();

         resultError("Error", "Error Removing the nature from the project");

      }
   }

   /**
    * Removes a nature from the project description if it already exists.
    * Requires that the nature be the first in the list.
    * 
    *
    * @param project  selected project resource is input for processing
    */
   public void removeNatureSpec(IProject project) throws CoreException {

      if (project.hasNature(IResourceIDs.NATURE_ID)) {

         // get project description and then the associated natures 
         IProjectDescription desp = project.getDescription();
         String[] natures = desp.getNatureIds();
         String[] newNatures = new String[natures.length - 1];
         int j = 0;
         for (int i = 0; i < natures.length; ++i) {
            if (!natures[i].equals(IResourceIDs.NATURE_ID)) {
               newNatures[j] = natures[i];
               break;
            } else {
               // Confirm nature removal
               resultInformation(
                  "Soln: Resource - Remove CustomNature Request",
                  "Custom Nature was removed.");
               break;
            }

         }

         desp.setNatureIds(newNatures);
         project.setDescription(desp, null);

      } else {
         // Identify Failure of Nature Removal
         // Logic should never be reached as the XML filters on the natureID
         resultError(
            "Soln: Resource - Remove CustomNature Request",
            "Custom Nature was not found, not able to remove.");

      }
   }

}