package com.ibm.lab.soln.resources.nature_builder;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */
 
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.runtime.CoreException;

import com.ibm.lab.soln.resources.ActionResourceAction;
import com.ibm.lab.soln.resources.IResourceIDs;

/**
 * Implements an action to add the <code>CustomNature</code> directly 
 * to an existing project. 
 * <p>
 * This demonstrates the API for the project and project description 
 * workspace resources and how you might implement this logic in your tool.
 * </p>
 * @see org.eclipse.core.resources.IProject
 * @see org.eclipse.core.resources.IProjectDescription
 * @see org.eclipse.core.resources.ICommand
 */

public class ActionAddCustomNature extends ActionResourceAction {

   /**
    * ActionAddBuilderToProject default constructor.
    */
   public ActionAddCustomNature() {
      super();
   }

   /**
    * Uses saved reference to selected project to call custom logic that
    * adds the builder to the current project.
    * 
    * @see org.eclipse.ui.IActionDelegate#run(IAction)
    * 
    * @param action action proxy that handles the presentation
    * portion of the plugin action
    */
   public void run(org.eclipse.jface.action.IAction action) {
      // Get project selected in Navigator
      // Extenstion definition filters to just single IProject selection
      IProject project =
         (IProject) this.selection.iterator().next();

      try {
         this.addCustomNature(project);
         // close and open project to tickle in the nature image
         project.close(null);
         project.open(null);

      } catch (CoreException e) {
         e.printStackTrace();

         // Problem with Builder Add
         resultError(
            "Soln: Resource - Add CustomNature Request",
            "Error adding the Custom Nature to the project.");

      }
   }
   /**
    * Add the nature to the project if it does not yet have the nature.
    * The process requires that you get the project description, 
    * get the current nature set, and then add the new nature to the set.
    * The first nature can control the nature image, so good form is to add 
    * a new nature to the end of the current list.
    */
   public void addCustomNature(IProject project)
      throws CoreException {

      if (!project.hasNature(IResourceIDs.NATURE_ID)) {
         try {
            IProjectDescription description =
               project.getDescription();
            String[] natures = description.getNatureIds();
            String[] newNatures = new String[natures.length + 1];
            System.arraycopy(
               natures,
               0,
               newNatures,
               0,
               natures.length);
            newNatures[natures.length] = IResourceIDs.NATURE_ID;
            description.setNatureIds(newNatures);
            project.setDescription(description, null);

            // Confirm Nature Add
            resultInformation(
               "Soln: Resource - Add CustomNature Request",
               "CustomNature added to the "
                  + project.getName()
                  + " project.");

         } catch (CoreException e) {
            e.printStackTrace();
            // Problem with Builder Add
            resultError(
               "Soln: Resource - Add CustomNature Request",
               "Error adding the CustomNature to the "
                  + project.getName()
                  + " project ");
         }

      } else {
         resultError(
            "Soln: Resource - Add CustomNature Request",
            "The CustomNature is already associated with the "
               + project.getName()
               + " project.");

      }
   }
}
