package com.ibm.lab.soln.resources;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.action.IAction;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * Multi-headed action where the processing depends on invocation point as 
 * determined by the UI action that triggered the request. Processing paths include:
 * <ul>
 * <li>ProjectDetails request - lists data for all projects. 
 * <li>ProjectStructure request - creates a folder/file structure and 
 * adds a BOOKMARK marker to the file.
 * <li>ProjectStructure request as runnable - the basic structure limits the number 
 * of events triggered by performing the workspace updates as part of an
 * <code>IWorkspaceRunnable</code>.
 * </ul>
 * <p>
 * Demonstrates the workspace get of a handle and existence checking with
 * respect to the workspace and event management using a workspace runnable.
 * </p>
 * @see org.eclipse.core.resources.IFile
 * @see org.eclipse.core.resources.IFolder
 * @see org.eclipse.core.resources.IProject
 * @see org.eclipse.core.resources.IMarker
 * @see org.eclipse.core.resources.IResource
 * @see org.eclipse.core.resources.IWorkspaceRoot
 * @see org.eclipse.core.resources.IWorkspaceRunnable
 */
public class ActionProcessResourceTree
  extends ActionResourceAction
  implements IViewActionDelegate, IWorkspaceRunnable {

  /** Name of Folder that will be created */
  private String imageFolderName = "images";

  /** Name of Folder that will be created */
  private String readmeFolderName = "readme";

  /** Name of txt File that will be created */
  private String txtFileName = "image_list.txt";

  /** Name of image file that will be created */
  private String imageFileName = "getstart_b.GIF";
  /**
   * ActionProcessResourceTree default constructor.
   */
  public ActionProcessResourceTree() {
  }
  /**
   * Uses saved reference to selected project to call custom logic that
   * processes the resource tree. Multiple plugin.xml defined actions map to 
   * this implementation.  The current action is detected and the appropriate 
   * logic is triggered based on the plugin.xml defined action id.
   * 
   * @see org.eclipse.ui.IActionDelegate#run(IAction)
   * 
   * @param action action proxy that handles the presentation
   * portion of the plugin action
   */
  public void run(IAction action) {
    // figure out what action logic to run based on xml id 
    if (action.getId().equals("projectDetails")) {
      showProjectDetails();
    } else {
      if (action.getId().equals("playPath")) {
        pathPlay();
      } else {
        if (action.getId().equals("RunnableAction")) {
          try {
            ResourcesPlugin.getWorkspace().run(this, null);
          } catch (CoreException e) {
            resultError(
              "Soln: Resource - Setup Project Structure Request",
              "Create project structure using a runnable failed.");
          }
        } else {
          // Assume project structure request Plugin definition means only a project can be passed 
          IProject project = (IProject) this.selection.iterator().next();
          createFolderFile(project);

        }
      }
    }
  }

  /**
   * @see org.eclipse.core.resources.IWorkspaceRunnable#run(IProgressMonitor)
   */
  public void run(IProgressMonitor monitor) throws CoreException {
    final IProject finalProject =
      (IProject) this.selection.iterator().next();
    createFolderFile(finalProject);
  }

  /**
   * Finds all projects in the workspace and produces a short list of 
   * project details (such as path location).
   */
  public void showProjectDetails() {

    // To navigate the workspace from a plug-in, we must first obtain the
    // IWorkspaceRoot, which represents the top of the resource hierarchy
    IWorkspaceRoot myWorkspaceRoot =
      ResourcesPlugin.getWorkspace().getRoot();
    IProject[] projList = myWorkspaceRoot.getProjects();

    // Create list of projects details to be shown
    String[] projects = new String[projList.length];
    String[] locations = new String[projList.length];
    for (int i = 0; i < projList.length; i++) {
      projects[i] = projList[i].toString();
      locations[i] = projList[i].getLocation().toString();
    }

    // Create project details dialog using workbench shell and project list
    ListDetailsDialog lDetails =
      new ListDetailsDialog(
        EDUResourcesPlugin
          .getDefault()
          .getWorkbench()
          .getActiveWorkbenchWindow()
          .getShell(),
        "Project Location Details",
        projects,
        "Projects:",
        locations,
        "Locations:");
    lDetails.open();

  }

  /**
   * Resource processing logic to add resources to a given project
   * one step at a time.  Steps are:
   * <ul>
   * <li>1.  Get folder named images. 
   * <li>1.a If folder does not exist, create it.
   * <li>1.b If folder exists, move on.
   * <p>
   * <li>2.  Get a file named image_list.txt in the new folder.
   * <li>2.a If file does not exist, create the file with predefined content.
   * <li>2.b If file exists, add a marker to the file.
   * <p>
   * <li>3.  Get a file named getstart_b.GIF in the new folder.
   * <li>3.a If file does not exist, create the file from a file in 
   * the plugin install directory.
   * <li>3.b If file exists, move on.
   * <p>
   * <li>4.  Get folder named images. 
   * <li>4.a If folder does not exist, create it.
   * <li>4.b If folder exists, move on.
   * <p>
   * <li>5.  Report results.
   * </ul>
   */
  public void createFolderFile(final IProject project) {

    boolean imageFolderAdded = false;
    boolean txtFileAdded = false;
    boolean imageFileAdded = false;
    boolean readmeFolderAdded = false;

    IFolder imageFolder = project.getFolder(imageFolderName);
    imageFolderAdded = createFolder(imageFolder);

    IFile newImageFile = imageFolder.getFile(imageFileName);
    if (!newImageFile.exists()) {
      newImageFile = copyBasics(imageFolder);
      if (newImageFile.exists())
        imageFileAdded = true;
    }

    IFile newTxtFile = imageFolder.getFile(txtFileName);
    if (!newTxtFile.exists()) {
      try {
        newTxtFile.create(getInitialContents(), false, null);
        txtFileAdded = true;
        addMarker(newTxtFile);
      } catch (CoreException ce) {
        ce.printStackTrace();
      };
    }

    IFolder readmeFolder = project.getFolder(readmeFolderName);
    readmeFolderAdded = createFolder(readmeFolder);

    // Create result message
    String createMsg = "Project Setup Results:";
    createMsg = createMsg.concat(folderStatus(imageFolder, imageFolderAdded));
    createMsg = createMsg.concat(fileStatus(newTxtFile, txtFileAdded));
    createMsg = createMsg.concat(fileStatus(newImageFile, imageFileAdded));
    createMsg = createMsg.concat(folderStatus(readmeFolder, readmeFolderAdded));

    // Show Project Setup Results
    resultInformation(
      "Soln: Resource - Setup Project Structure Request",
      createMsg);

  }
  /**
   * Creates a folder if it does not exist, returns true if a folder
   * was created.
   * 
   * @param newFolder
   * @return boolean
   */
  public boolean createFolder(IFolder newFolder) {
    if (!newFolder.exists()) {
      try {
        newFolder.create(true, true, null);
      } catch (CoreException ce) {
        ce.printStackTrace();
        return false;
      };
    } else {
      return false;
    };
    return true;
  }

  /**
   * Returns string that describes if the folder was 
   * created or already existed.
   * 
   * @param imageFolder
   * @param imageFolderAdded
   * @return String
   */
  public String folderStatus(
    IFolder imageFolder,
    boolean imageFolderAdded) {
    String createMsg = null;
    if (imageFolderAdded) {
      createMsg = "\n\t> + folder added: " + imageFolder.getName();
    } else {
      createMsg =
        "\n\t> = folder already existed: " + imageFolder.getName();
    }
    return createMsg;
  }
  /**
   * Returns string that describes if the file was 
   * created or already existed.
   * 
   * @param txtFile
   * @param txtFileAdded
   * @return String
   */
  public String fileStatus(IFile txtFile, boolean txtFileAdded) {
    String createMsg = null;
    if (txtFileAdded) {
      createMsg = "\n\t> + file added: " + txtFile.getName();
    } else {
      createMsg = "\n\t> = file already existed: " + txtFile.getName();
    }
    return createMsg;
  }
  /**
   * Return input stream used to create initial file contents. 
   */
  public InputStream getInitialContents() {
    StringBuffer sb = new StringBuffer();
    sb.append("image_list.txt\n");
    sb.append("A good place to describe your images\n\n");
    sb.append("Image File name \t\t Description\n");
    sb.append("--------------- \t\t -----------\n");
    sb.append("getstart_b.GIF  \t\t glowing lamp image\n ");
    sb.append("        \n ");

    return new ByteArrayInputStream(sb.toString().getBytes());
  }

  /**
   * Add marker to identified resource. 
   * Bookmark marker is added using predefined <code>CHAR_START</code>,
   * <code>CHAR_END</code>, and <code>MESSAGE</code> values.
   */
  public void addMarker(IResource resource) {
    try {
      IMarker marker = resource.createMarker(IMarker.BOOKMARK);
      marker.setAttribute(IMarker.CHAR_START, 153);
      marker.setAttribute(IMarker.CHAR_END, 154);
      marker.setAttribute(IMarker.MESSAGE, "New " + txtFileName + " added");
    } catch (CoreException e) {
      resultError(
        "Soln: Resource - Marker Create",
        "Creation of marker failed.");
    }
  }
  /**
   * Creates a file in the passed folder using content copied from
   * a standard file found in the plug-in's install directory. 
   * 
   * @param folder - folder where standard image file should be created
   */
  public IFile copyBasics(IFolder folder) {

    // create a file folder resource tree
    IFile imageFile = folder.getFile(imageFileName);

    // create the imageFile content using a file from the plugin install directory
    try {
      imageFile.create(
        EDUResourcesPlugin.getDefault().openStream(
          new Path(imageFolderName + "/" + imageFileName)),
        true,
        null);

    } catch (CoreException e) {
      e.printStackTrace();
    } catch (IOException e) {
      e.printStackTrace();
    }
    return imageFile;
  }

  /**
   * Produces a list of path values based on the selected resource.
   * 
   */
  public void pathPlay() {
    IResource res = (IResource) selection.getFirstElement();
    String pathList = new String();
    pathList = pathList = pathList.concat("Resource used: " + res + "\n");

    traceEnabled = false;

    // Create list of projects details to be shown
    String[] methods = new String[7];
    String[] values = new String[7];

    methods[0] = "res.getLocation.toString()";
    values[0] = res.getLocation().toString();

    methods[1] = "res.getFullPath().toString()";
    values[1] = res.getFullPath().toString();

    methods[2] = "res.getFullPath().toOSString()";
    values[2] = res.getFullPath().toOSString();

    methods[3] = "res.getProjectRelativePath().toString()";
    values[3] = res.getProjectRelativePath().toString();

    // absolute/relative
    methods[4] = "res.getProjectRelativePath().makeAbsolute().toString()";
    values[4] = res.getProjectRelativePath().makeAbsolute().toString();

    methods[5] = "res.getFullPath().makeRelative().toString()";
    values[5] = res.getFullPath().makeRelative().toString();

    methods[6] = "res.getFullPath().makeUNC(true).toString()";
    values[6] = res.getFullPath().makeUNC(true).toString();

    // Create project details dialog using workbench shell and project list
    ListDetailsDialog lDetails =
      new ListDetailsDialog(
        EDUResourcesPlugin
          .getDefault()
          .getWorkbench()
          .getActiveWorkbenchWindow()
          .getShell(),
        "Path Value List for: " + res,
        methods,
        "Path Method:",
        values,
        "Path Value:");
    lDetails.open();

  }

  /**
   * Not used in this action - implemented per 
   * <code>IViewActionDelegate</code> interface.
   * 
   * @see org.eclipse.ui.IViewActionDelegate#init(IViewPart)
   */
  public void init(IViewPart view) {
  }

}