package com.ibm.lab.soln.dialogs;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;

/**
 * Demonstration of a simple multi-page wizard that supports controlled paging and file creation.
 * 
 * Three pages are added in the addPages() method.
 * 
 * File creation is supported through a custom page that inherits 
 * container selection and file name validation.
 * 
 * Additional pages support testing of the page complete status using values 
 * passed to the wizard page constructors.
 * 
 * @see org.eclipse.jface.wizard.Wizard
 * @see org.eclipse.ui.INewWizard
 */
public class StructureWizard
  extends Wizard
  implements INewWizard {
  // switch to control write of trace data
  private boolean traceEnabled = true;
  private boolean workState = true;


  // to hold selected project resource for run method access
  private IStructuredSelection selection;

  private HashMap resourceNames;

  // wizard page fields to support references in wizard logic
  private StructureInitPage selPage;

  /**
   * The constructor.
   */
  public StructureWizard() {
  }

  /**
   * Passes the wizard finish request on to each resource creation page.
   * Most pages will have been finished already, as the <code>getNextPage</code>
   * logic in each page calls the page <code>finish</code> as they page forward.
   * The last page added will still need to receive a <code>finish</code> request.
   * <p>
   * Sending the <code>finish</code> request to a page more than once is ok, each 
   * page caches the resource after initial creation and returns the original 
   * instance.
   * 
   * @see Wizard#performFinish
   */
  public boolean performFinish() {

    IWizardPage[] pageSet = getPages();

    workState = true;
    // skip first page - not a resource creation page
    for (int i = 1; i < pageSet.length; i++) {

      if (pageSet[i]
        instanceof StructureNewFolderWizardPage) {
        ((StructureNewFolderWizardPage) pageSet[i]).finish();
      } else {
        if (pageSet[i]
          instanceof StructureNewFileWizardPage) {
          ((StructureNewFileWizardPage) pageSet[i]).finish();
        }
      }
    }
    return workState;
  }

  /**
   * Initialize processing saves local workbench and selection references and sets wizard title/image.
   */
  public void init(
    IWorkbench workbench,
    IStructuredSelection selection) {
    // tracing output
    traceMsg("MBWizard: ----> Start Wizard");
    traceMsg("MBWizard: -------> in method init.");
    // Wizard customization
    setWindowTitle("Project Structure Wizard");
    setDefaultPageImageDescriptor(
      getImageDescriptor("eclipse.jpg"));
    // Saving local references to the workbench and resource selected in navigator
    this.selection = selection;
    //      this.myWorkbench = workbench;
  } 
  /**
   * Utility method that returns a named image from the plugin icons directory.
   * @param relativePath  name or directory/name of the image requested
   * @return  an ImageDescriptor for the identified image if found at the path
   */
  private ImageDescriptor getImageDescriptor(String relativePath) {
    String iconPath = "icons/";
    try {
      DialogsPlugin plugin = DialogsPlugin.getDefault();
      URL installURL = plugin.getDescriptor().getInstallURL();
      URL url = new URL(installURL, iconPath + relativePath);
      return ImageDescriptor.createFromURL(url);
    } catch (MalformedURLException e) {
      // Should not happen
      return null;
    }
  } 
  /**
   * Add predetermined pages that are required in the wizard.
   * Other pages could be added after the wizard starts using the addPage() method.
   * 
   * @see org.eclipse.jface.wizard.Wizard#addPages()
   */
  public void addPages() {
    // Only adding one page, but need the next button
    setForcePreviousAndNextButtons(true);

    // Add starter page to the wizard page sequence
    selPage =
      new StructureInitPage("SelPage", selection, this);
    addPage(selPage);
  } 
  /**
   * Write trace statements.  
   * System.out.println with prefix tagging used for simplicity.
   */
  void traceMsg(String msg) {
    if (IDialogsIDs.TRACE_ENABLED | traceEnabled)
      System.out.println(msg);
  }

  /**
   * Used to show action results.
   * 
   * @see org.eclipse.jface.dialogs.MessageDialog
   */
  private void resultInformation(
    String title,
    String msg) {
    // Indicate Error
    MessageDialog.openInformation(null, title, msg);
  }

}