package com.ibm.lab.soln.dialogs;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.IFile;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.dialogs.WizardNewFileCreationPage;

/**
 * Wizard page designed to create new file resources as part of a continuous 
 * flow multi-page wizard as implemented by the <code>StructureWizard</code>.
 * Also allows for selection of the next resource creation page, by type.
 * <p>
 * Demonstrates dynamic addition and control of wizard pages in a wizard.
 * </p>
 */
public class StructureNewFileWizardPage
  extends WizardNewFileCreationPage
  implements SelectionListener {

  // switch to control write of trace data
  private boolean traceEnabled = true;

  private Button folderButton; // folder
  private Button fileButton; // file
  private Button noneButton; // none

  private Button disablePage;

  private Composite resourceControl;

  private boolean nextPageExists;
  private IWizardPage nextPage = null;

  private boolean pastFirstEntry = false;

  private IStructuredSelection selection;
  private Wizard currentWizard;

  /**
   * Constructor for StructureNewFileWizardPage.
   * @param pageName
   * @param selection
   */
  public StructureNewFileWizardPage(
    String pageName,
    IStructuredSelection selection,
    Wizard parent) {
    super(pageName, selection);
    this.selection = selection;
    currentWizard = parent;

    setDescription("Create a new file, or if you changed your mind, choose to ignore the page and choose another resource type.");
    setTitle("Project Structure: File Creation");
  }

  /**
   * Creates the resource if the page has not been disabled. May be called as the 
   * next page is requested, or by the wizard <code>finish()</code> method.
   */
  public boolean finish() {
    // create the new file resource
    IFile newFile = null;

    if (!disablePage.getSelection()) {
      newFile = createNewFile();
      disablePage.setEnabled(false);
    }
    if (newFile == null)
      return false; //creation was unsuccessful

    return true;
  }

  /**
   * Creates the top level dialog control under the given parent composite.
   * <p>
   * Implementors are responsible for ensuring that the created control can 
   * be accessed via <code>getControl</code>. This is done by identifying the main 
   * control you create using the <code>setControl</code> method.
   * </p>
   *
   * @see org.eclipse.jface.dialogs.IDialogPage#createControl(Composite)
   * 
   * @param parent the parent composite
   */
  public void createControl(Composite parent) {

    Composite composite = new Composite(parent, SWT.NONE);

    GridLayout layout = new GridLayout();
    layout.numColumns = 1;
    composite.setLayout(layout);

    super.createControl(composite);

    // get control super identified before replacing
    resourceControl = (Composite) getControl();

    Label space1 = new Label(composite, SWT.SEPARATOR | SWT.HORIZONTAL);
    GridData data1 = new GridData(GridData.FILL_BOTH);
    space1.setLayoutData(data1);

    disablePage = new Button(composite, org.eclipse.swt.SWT.CHECK);
    disablePage.setText("Select to ignore this resource creation page");
    disablePage.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent event) {
        if (((Button) event.widget).getSelection())
          setPageComplete(true);
        else
          setPageComplete(validatePage());
        //
        getWizard().getContainer().updateButtons();
      }
    });

    Label labelT = new Label(composite, SWT.NONE);
    labelT.setText(
      "Note: Unless ignored, resource creation occurs as you page forward.");

    Label space2 = new Label(composite, SWT.SEPARATOR | SWT.HORIZONTAL);
    GridData data2 = new GridData(GridData.FILL_BOTH);
    space2.setLayoutData(data2);

    // add group control for the next page choice
    createNextPageSelection(composite);

    // redefine main control to local parent (replaces superclass defined main control)
    setControl(composite);

  }

  /**
   * Utility method that creates a group with radio buttons that support user 
   * selection of the next create resource wizard page.
   *
   * @param composite  the parent for the new controls
   */
  private void createNextPageSelection(Composite composite) {
    Group group = new Group(composite, SWT.NONE);
    group.setText("Select Next Project Structure Page:");
    GridData gData = new GridData(GridData.FILL_HORIZONTAL);
    group.setLayoutData(gData);

    GridLayout gLayout = new GridLayout();
    gLayout.numColumns = 1;
    group.setLayout(gLayout);

    Label labelN = new Label(group, SWT.NONE);
    labelN.setText("Select the resource to create next or none to stop:");
    // Add Page Control 

    folderButton = createRadioButton(group, "Create another fo&lder");
    fileButton = createRadioButton(group, "Create another f&ile");
    noneButton = createRadioButton(group, "N&o more Resources to create");
  }

  /**
   * Utility method that creates a radio button instance
   * and sets the default layout data.
   *
   * @param parent  the parent for the new button
   * @param label  the label for the new button
   * @return the newly-created button
   */
  private Button createRadioButton(Composite parent, String label) {
    Button button = new Button(parent, SWT.RADIO | SWT.LEFT);
    button.setText(label);
    GridData data = new GridData(GridData.FILL_HORIZONTAL);
    button.setLayoutData(data);
    button.addSelectionListener(this);
    return button;
  }

  /* non-Javadoc
   * 
   * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(SelectionEvent)
   */
  public void widgetDefaultSelected(SelectionEvent e) {
  }

  /**
   * Identifies if another resource creation page has been requested in the 
   * wizard flow.  Choice is flexible until the page has been entered.
   * 
   * @see org.eclipse.swt.events.SelectionListener#widgetSelected(SelectionEvent)
   */
  public void widgetSelected(SelectionEvent e) {
    if (folderButton.getSelection()) {
      System.out.println("new folder request");
      nextPageExists = true;

    } else {
      if (fileButton.getSelection()) {
        System.out.println("new file request");
        nextPageExists = true;
      } else {
        if (noneButton.getSelection()) {
          System.out.println("no more request");
          nextPageExists = false;
        }

      }
    }
    validatePage();
    getWizard().getContainer().updateButtons();
  }

  /**
   * Alternate version of the <code>canFlipToNextPage()</code> method that only
   * allows the Next button to be enabled if another resource creation page has 
   * been selected.  This is determined by the selection event listener logic
   * for the buttons added in <code>createControl()</code>.
   * 
   * @see org.eclipse.jface.wizard.IWizardPage#canFlipToNextPage()
   */
  public boolean canFlipToNextPage() {
    return nextPageExists;
  }

  /**
   * Alternate version of the <code>getNextPage()</code> method that adds the 
   * selected new resource page as the next page.  Once created and added to the
   * wizard, the next page selection options are disabled. The page can be disabled,
   * but not removed from the current page set.
   * 
   * @see org.eclipse.jface.wizard.IWizardPage#getNextPage()
   */
  public IWizardPage getNextPage() {
    System.out.println("In getNextPage");
    if (nextPage == null) {
      if (folderButton.getSelection()) {
        nextPage =
          new StructureNewFolderWizardPage(
            "fold",
            selection,
            currentWizard);
      } else {
        if (fileButton.getSelection()) {
          nextPage =
            new StructureNewFileWizardPage(
              "file",
              selection,
              currentWizard);
        }
      }
      currentWizard.addPage(nextPage);
      folderButton.setEnabled(false);
      fileButton.setEnabled(false);
      noneButton.setEnabled(false);
    }
    // If page not to be skipped for processing
    if (!disablePage.getSelection())
      // once resource can be created, then protect resource spec
      if (validatePage()) {
        resourceControl.setEnabled(false);
        // if resource created ok, then identify in page description
        if (finish())
          this.setDescription("Resource created");
      }
    return super.getNextPage();
  }

  /**
   * Reworked validate page logic forces true when page is to be ignored. This 
   * allows for invalid or missing resource names to be ignored when they will
   * not be processed.
   * 
   * @see org.eclipse.ui.dialogs.WizardNewFolderMainPage#validatePage()
   */
  protected boolean validatePage() {
    if ((disablePage != null) && (!disablePage.getSelection()))
      return super.validatePage();
    else
      return true;
  }

  /**
   * Only validate resource name entry after first entry, so as not to block page 
   * description during initial display.
   * 
   * @see org.eclipse.jface.dialogs.IDialogPage#setVisible(boolean)
   */
  public void setVisible(boolean visible) {
    if (pastFirstEntry)
      validatePage();
    pastFirstEntry = true;
    super.setVisible(visible);
  }

  /**
   * Write trace statements.  
   * System.out.println with prefix tagging used for simplicity.
   */
  private void traceMsg(String msg) {
    if (IDialogsIDs.TRACE_ENABLED | traceEnabled)
      System.out.println(msg);
  }

}