package com.ibm.lab.soln.dialogs;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPropertyPage;
import org.eclipse.ui.dialogs.PropertyPage;

/**
 * Displays two properties, <pre>My_Prop_Name</pre> and <pre>Recent_Edits</pre> 
 * associated with the selected resource. These properties are stored as 
 * persistent properties. Keys are defined in <code>IDialogsIDs</code>.
 * 
 * @see com.ibm.lab.soln.dialogs.IDialogsIDs
 * @see org.eclipse.ui.dialogs.PropertyPage
 * @see org.eclipse.ui.IWorkbenchPropertyPage
 */
public class MyPropertyPage
  extends PropertyPage
  implements IWorkbenchPropertyPage {

  // switch to control write of trace data
  private boolean traceEnabled = false;

  // widgets used in property page to define property values
  private Text textInfo;
  private Button reditFlag;

  /**
   * Default constructor.
   */
  public MyPropertyPage() {
  }

  /**
   * Creates the user interface for the property page. The property value for
   * the <pre>Recent_Edits</pre> key is obtained at this time.  The 
   * <pre>My_Prop_Name</pre> property key value is obtained in the 
   * <code>setVisible()</code> method.
   * 
   * @see org.eclipse.jface.preference.PreferencePage#createContents
   */
  protected Control createContents(Composite parent) {

    Composite composite = new Composite(parent, SWT.NONE);
    GridLayout compositeLayout = new GridLayout();
    compositeLayout.numColumns = 1;
    compositeLayout.horizontalSpacing = 10;
    compositeLayout.verticalSpacing = 10;
    compositeLayout.marginHeight = 10;
    compositeLayout.marginWidth = 10;
    composite.setLayout(compositeLayout);

    GridData compositeData = new GridData(GridData.FILL_BOTH);
    compositeData.grabExcessHorizontalSpace = true;
    composite.setLayoutData(compositeData);

    Group group = new Group(composite, SWT.NONE);
    group.setText("Recent Edits Setting");
    
    GridData groupData = new GridData(GridData.GRAB_HORIZONTAL);
    group.setLayoutData(groupData);

    GridLayout groupLayout = new GridLayout();
    groupLayout.numColumns = 1;
    groupLayout.horizontalSpacing = 15;
    groupLayout.verticalSpacing = 10;
    groupLayout.marginHeight = 15;
    groupLayout.marginWidth = 15;
    group.setLayout(groupLayout);

    Label labelB = new Label(group, SWT.NONE);
    labelB.setText(
      "Select to disable Recent Edits list support:");

    reditFlag = new Button(group, SWT.CHECK);
    reditFlag.setText(
      "Exclude from Recent Edit Marker Support");
    reditFlag.setSelection(getReditPropertyState());

    Label labelC = new Label(group, SWT.NONE);
    labelC.setText(
      "Note: Recent Edits support is provided by the \n"
        + "com.ibm.lab.soln.resources plug-in.");

    Label labelT = new Label(composite, SWT.NONE);
    labelT.setText(
      "Enter value to be saved with the resource:");

    textInfo = new Text(composite, SWT.SINGLE | SWT.BORDER);
    // content defined in setVisible
    GridData textData = new GridData(GridData.FILL_HORIZONTAL);
    textInfo.setLayoutData(textData);

    return composite;
  }

  /**
   * Method triggered as page is entered and exited. In this demonstration page
   * the <pre>My_Prop_Name</pre> property key value is obtained every time 
   * the page is entered.  The other key shown on this page is only retrieved
   * as the user interface is constructed in the <code>createContents()</code>
   * method.
   * 
   * @see org.eclipse.jface.dialogs.DialogPage#setVisible
   */
  public void setVisible(boolean visible) {

    super.setVisible(visible);

    traceMsg(
      "PropCycle: " + "> in setVisible(" + visible + ")");

    IResource resource = (IResource) getElement();

    try {
      String propValue =
        resource.getPersistentProperty(
          IDialogsIDs.PROP_NAME_KEY);
      if (propValue != null)
        textInfo.setText(propValue);
      else
        textInfo.setText("No prop value found");
    } catch (CoreException e) {
      e.printStackTrace();
    }
  }

  /**
   * Can be used to implement any special processing, such as notification,
   * if required. Logic to actually change properties should be in the
   * <code>performOk</code> method as that method will also be triggered 
   * when the Apply push button is selected.
   * 
   * @see org.eclipse.jface.preference.PreferencePage#performApply
   */
  protected void performApply() {
    super.performApply();
    traceMsg("PropCycle: " + "> in performApply");
  }

  /**
   * Can be used to implement any cleanup processing that may be required.
   * 
   * @see org.eclipse.jface.preference.PreferencePage#performCancel
   */
  public boolean performCancel() {
    super.performCancel();
    traceMsg("PropCycle: " + "> in performCancel");
    return true;
  }

  /**
   * Can be used to retrieve and set any defaults for the properties that
   * are displayed as part of this page.
   * 
   * @see org.eclipse.jface.preference.PreferencePage#performDefaults
   */
  protected void performDefaults() {
    super.performDefaults();
    traceMsg("PropCycle: " + "> in performDefaults");
  }

  /**
   * Saves a persistent property for the current resource. This method
   * is triggered when either the Apply or Ok push button has been selected.
   * 
   * @see org.eclipse.jface.preference.PreferencePage#performOk
   */
  public boolean performOk() {
    traceMsg("PropCycle: " + "> in performOk");
    IResource resource = (IResource) getElement();
    try {
      resource.setPersistentProperty(
        IDialogsIDs.PROP_NAME_KEY,
        textInfo.getText());
      setReditPropertyState(reditFlag.getSelection());
    } catch (CoreException e) {
      e.printStackTrace();
    }
    return super.performOk();
  }

  /**
   * Gets the property value for the <pre>Recent_Edits</pre> key.
   * 
   * @return boolean
   */
  public boolean getReditPropertyState() {

    IResource resource = (IResource) getElement();

    try {
      String propValue =
        resource.getPersistentProperty(
          IDialogsIDs.REDIT_PROPERTY_KEY);

      if ((propValue != null) && propValue.equals("True"))
        return true;
      else
        return false;
    } catch (CoreException e) {
      // deal with exception

    }
    return false;
  }

  /**
   * Sets the property value for the <pre>Recent_Edits</pre> key.
   * 
   */
  public void setReditPropertyState(boolean value) {
    IResource resource = (IResource) getElement();

    try {
      if (value) {
        resource.setPersistentProperty(
          IDialogsIDs.REDIT_PROPERTY_KEY,
          "True");
      } else {
        resource.setPersistentProperty(
          IDialogsIDs.REDIT_PROPERTY_KEY,
          "False");
      }
    } catch (CoreException e) {
    }

  }

  /**
   * Write trace statements.  
   * System.out.println used for simplicity.
   */
  private void traceMsg(String msg) {
    if (IDialogsIDs.TRACE_ENABLED | traceEnabled)
      System.out.println(msg);
  }
}