package com.ibm.lab.soln.dialogs;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPluginDescriptor;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.plugin.AbstractUIPlugin;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * The plugin class for the Dialogs Plugin. Extends the Eclipse plug-in support
 * provided by <code>AbstractUIPlugin</code>.
 * Instantiated by the platform when the plug-in is started.
 */
public class DialogsPlugin extends AbstractUIPlugin {
  //Reference to the shared instance, returned when requested.
  private static DialogsPlugin plugin;
  //Resource bundle.
  private ResourceBundle resourceBundle;

  // switch to control write of trace data
  private boolean traceEnabled = false;

  /**
   * The constructor.
   */
  public DialogsPlugin(IPluginDescriptor descriptor) {
    super(descriptor);
    plugin = this;
    try {
      resourceBundle =
        ResourceBundle.getBundle(
          "com.ibm.lab.soln.dialogs.DialogsPluginResources");
    } catch (MissingResourceException x) {
      resourceBundle = null;
    }
  }

  /**
    * Sets the default preference values for the plugin.
    * When this method is used the IPreferenceStore object 
    * is provided as a parameter.
    * 
    * This method is triggered when the preferences for the plugin are first
    * requested.  This is the only place default settings are defined.
    * 
    * The alternative method, <code>initializeDefaultPluginPreferences()</code>
    * can be used, but you would then need to get the Preferences object for the 
    * Plugin. You need to use one or the other, but not both.  
    * 
    * @see org.eclipse.ui.plugin.AbstractUIPlugin#initializeDefaultPreferences
    */
  protected void initializeDefaultPreferences(IPreferenceStore store) {
    // Set default for string value
    store.setDefault("text_field_key", "myTextDefaultValue");
    // Limit for number of entries to be kept as RecentEdits markers
    store.setDefault(
      IDialogsIDs.REDIT_PREFERENCES_KEY,
      IDialogsIDs.REDIT_PREFERENCES_DEFAULT);
      store.setDefault(IDialogsIDs.REDIT_PREFERENCES_KEY2, true);

    traceMsg("set preference defaults");
  }

  /**
   * Returns the shared instance.
   */
  public static DialogsPlugin getDefault() {
    return plugin;
  }

  /**
   * Returns the workspace instance.
   */
  public static IWorkspace getWorkspace() {
    return ResourcesPlugin.getWorkspace();
  }

  /**
   * Returns the string from the plugin's resource bundle,
   * or 'key' if not found.
   */
  public static String getResourceString(String key) {
    ResourceBundle bundle = DialogsPlugin.getDefault().getResourceBundle();
    try {
      return bundle.getString(key);
    } catch (MissingResourceException e) {
      return key;
    }
  }

  /**
   * Returns the plugin's resource bundle,
   */
  public ResourceBundle getResourceBundle() {
    return resourceBundle;
  }
  /**
   * Write trace statements.  
   * System.out.println with prefix tagging used for simplicity.
   */
  private void traceMsg(String msg) {
    if (traceEnabled)
      System.out.println(msg);
  }

  /**
   * Logic to manage the addition and removal of a change
   * listener for preference settings.  Used by the 
   * <code>MyPrefPageControl</code> preference page to control when
   * a change listener is active.
   * 
   */
  public static class PreferenceListener {
    // fields for the preference change listener options
    private static boolean hasPrefListener = false;
    private static IPropertyChangeListener prefStoreCL;
    private static Preferences.IPropertyChangeListener preferencesCL;

    // switch for preference setting API choice
    // true uses PreferenceStore, false uses Preferences
    private static boolean prefStore = false;

    /**
     * Returns the hasPrefListener.
     * @return boolean
     */
    public static boolean hasPrefListener() {
      return hasPrefListener;
    }

    /**
     * Add a change listener for <code>DialogsPlugin</code> managed 
     * preferences. Two options are available, both are defined, 
     * but only one is invoked here, depending on the value of
     * the <code>hasPrefListener</code> field.
     */
    public static void addListener() {
      if (prefStore)
        // This method adds a listener using the PreferenceStore API
        addPrefStoreListener();
      else
        // This method adds a listener using the Preferences API
        addPreferencesListener();

      hasPrefListener = true;

      resultInformation(
        "Add Property Change Listener Request",
        "Preferences change listener added. "
          + "\nEach change made to a preference setting managed by the Dialogs plug-in "
          + "will be reported.");
    }

    /**
     * Adds a preferences listener, using the listener and event types
     * defined in the <code>Preferences</code> class.
     */
    public static void addPreferencesListener() {
      // create preferences change listener
      preferencesCL = new Preferences.IPropertyChangeListener() {
        public void propertyChange(Preferences.PropertyChangeEvent event) {

          resultInformation(
            "<Got told something by a PreferenceStore object>",
            "Preference property: "
              + event.getProperty()
              + "\n\told setting: "
              + event.getOldValue()
              + "\n\tnew setting: "
              + event.getNewValue());
        };
      };
      // associate listener with Preferences object
      DialogsPlugin
        .getDefault()
        .getPluginPreferences()
        .addPropertyChangeListener(
        preferencesCL);
    }

    /**
     * Adds a PreferenceStore listener using the <code>IPropertyChangeListener</code>
     * listener and <code>PropertyChangeEvent</code> event types.
     */
    public static void addPrefStoreListener() {
      // create preference store change listener
      prefStoreCL = new IPropertyChangeListener() {
        public void propertyChange(PropertyChangeEvent event) {

          resultInformation(
            "<Got told something by a Preferences object>",
            "Preference property: "
              + event.getProperty()
              + "\n\told setting: "
              + event.getOldValue()
              + "\n\tnew setting: "
              + event.getNewValue());
        };
      };
      // associate listener with PreferenceStore object
      DialogsPlugin
        .getDefault()
        .getPreferenceStore()
        .addPropertyChangeListener(
        prefStoreCL);
    }

    /**
     * Removes the change listener that was added previously.
     */
    public static void removeListener() {
      if (prefStore)
        DialogsPlugin
          .getDefault()
          .getPreferenceStore()
          .removePropertyChangeListener(prefStoreCL);
      else
        DialogsPlugin
          .getDefault()
          .getPluginPreferences()
          .removePropertyChangeListener(preferencesCL);

      hasPrefListener = false;
      resultInformation(
        "Remove Property Change Listener Request",
        "Preferences change listener removed. ");
    }

    /**
     * Uses a <code>MessageDialog</code> to show preference value changes.
     * 
     * @see org.eclipse.jface.dialogs.MessageDialog
     */
    protected static void resultInformation(String title, String msg) {
      // Confirm Result
      Shell shell =
        DialogsPlugin
          .getDefault()
          .getWorkbench()
          .getActiveWorkbenchWindow()
          .getShell();
      MessageDialog.openInformation(shell, title, msg);
    }

  }
}