package com.ibm.lab.soln.msseditor.ui;

/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.text.MessageFormat;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.dialogs.SaveAsDialog;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.part.FileEditorInput;

import com.ibm.lab.soln.msseditor.core.IMiniSpreadsheetListener;
import com.ibm.lab.soln.msseditor.core.MiniSpreadsheet;
import com.ibm.lab.soln.msseditor.core.MiniSpreadsheetRow;
import com.ibm.lab.soln.msseditor.ui.actions.AppendRowAction;
import com.ibm.lab.soln.msseditor.ui.actions.RemoveRowAction;

//Edu-Sol : 04 Editor defined
public class MiniSpreadsheetEditor
  extends EditorPart
  implements IMiniSpreadsheetListener, IResourceChangeListener {

  private TableViewer tableViewer;
  private Table table;
  private MiniSpreadsheet miniSpreadsheet =
    new MiniSpreadsheet(
      MiniSpreadsheet.DEFAULT_ROW_COUNT,
      MiniSpreadsheet.DEFAULT_COLUMN_COUNT);
  private boolean isDirty = false;
  static final String STD_HEADINGS = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
  private int defaultAlignment;
  private Menu contextMenu;

  public MiniSpreadsheetEditor() {
  }

  /**
   * @see org.eclipse.ui.IWorkbenchPart#createPartControl(Composite)
   */
  public void createPartControl(Composite parent) {
  	 
    createTable(parent);
    tableViewer = new TableViewer(table);
    getSite().setSelectionProvider(tableViewer);
    initializeTableLayout(parent);

    createColumns();
    initializeCellEditors();
    createContextMenu();
//	Edu-Sol : 06,7,8 Create Editor UI (viewer, content/label provider, input)
    tableViewer.setContentProvider(new MiniSpreadsheetContentProvider());
    tableViewer.setLabelProvider(new MiniSpreadsheetLabelProvider());
    tableViewer.setInput(miniSpreadsheet);
  }

  public void initializeTableLayout(Composite parent) {
    parent.setLayout(new FormLayout());

    FormData formData = new FormData();
    formData.left = new FormAttachment(0, 5);
    formData.right = new FormAttachment(100, -5);
    formData.top = new FormAttachment(0, 5);
    formData.bottom = new FormAttachment(100, -5);
    table.setLayoutData(formData);
  }

  private void initializeCellEditors() {
    int columnCount = table.getColumnCount();
    CellEditor editors[] = new CellEditor[columnCount];
    String[] properties = new String[columnCount];

    for (int i = 1; i < columnCount; i++) {
      properties[i] = Integer.toString(i);
      editors[i] = new TextCellEditor(table);
    }

    tableViewer.setColumnProperties(properties);
    tableViewer.setCellEditors(editors);
    tableViewer.setCellModifier(new ICellModifier() {
      public void modify(Object element, String property, Object value) {
        MiniSpreadsheetRow msr;
        if (element instanceof Item)
          msr = (MiniSpreadsheetRow) ((Item) element).getData();
        else
          msr = (MiniSpreadsheetRow) element;

        int columnIndex = Integer.parseInt(property) - 1;
        miniSpreadsheet.setData(
          msr.getRowIndex(),
          columnIndex,
          (String) value);
      }

      public boolean canModify(Object element, String property) {
        return true;
      }

      public Object getValue(Object element, String property) {
        MiniSpreadsheetRow msr = (MiniSpreadsheetRow) element;
        int columnIndex = Integer.parseInt(property) - 1;
        return msr.getString(columnIndex);
      }
    });
  }

  private void createTable(Composite parent) {
    table =
      new Table(
        parent,
        SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI | SWT.FULL_SELECTION);
    table.setLinesVisible(true);
  }

  private void createColumns() {
    TableLayout layout = new TableLayout();
    table.setLayout(layout);
    table.setHeaderVisible(true);
    TableColumn rn = new TableColumn(table, SWT.NONE);
    rn.setResizable(false);
    rn.setAlignment(SWT.CENTER);
    layout.addColumnData(new ColumnWeightData(5, 40, false));

    for (int i = 0; i < miniSpreadsheet.getColumnCount(); i++) {
      layout.addColumnData(new ColumnWeightData(10));
      TableColumn tc = new TableColumn(table, SWT.NONE);
      if (i < STD_HEADINGS.length())
        tc.setText(new Character(STD_HEADINGS.charAt(i)).toString());
      else
        tc.setText(Integer.toString(i + 1));

      final TableColumn tc2 = tc;
      tc2.addSelectionListener(new SelectionListener() {
        public void widgetSelected(SelectionEvent e) {
          table.setRedraw(false);
          table.setHeaderVisible(false);
          if (tc2.getAlignment() == SWT.LEFT)
            tc2.setAlignment(SWT.CENTER);
          else if (tc2.getAlignment() == SWT.CENTER)
            tc2.setAlignment(SWT.RIGHT);
          else
            tc2.setAlignment(SWT.LEFT);
          table.setHeaderVisible(true);
          table.setRedraw(true);
        }
        public void widgetDefaultSelected(SelectionEvent e) {
        }
      });
    }
    setAlignment(SWT.CENTER);
  }
//Edu-Sol : 13 Add Editor actions
  private void createContextMenu() {
    MenuManager menuMgr =
      new MenuManager(getEditorSite().getId(), getSite().getId());
    menuMgr.setRemoveAllWhenShown(true);

    menuMgr.addMenuListener(new IMenuListener() {
      public void menuAboutToShow(IMenuManager manager) {
        MiniSpreadsheetEditor.this.fillContextMenu(manager);
      }
    });

    Menu contextMenu = menuMgr.createContextMenu(table);
    table.setMenu(contextMenu);

    getSite().registerContextMenu(
      menuMgr,
      getSite().getSelectionProvider());
  }

  private void fillContextMenu(IMenuManager menuMgr) {
    AppendRowAction appendRowAction = new AppendRowAction();
    appendRowAction.setActiveEditor(this);
    menuMgr.add(appendRowAction);

    menuMgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));

    RemoveRowAction removeRowAction = new RemoveRowAction();
    removeRowAction.setActiveEditor(this);
    menuMgr.add(removeRowAction);
  }

  /**
   * Passing the focus request to the viewer's control.
   */
  public void setFocus() {
    table.setFocus();
  }

  /**
   * @see org.eclipse.ui.IEditorPart#doSave(IProgressMonitor)
   */
//Edu-Sol : 10a Save Editor content
  public void doSave(IProgressMonitor monitor) {
    try {
      IFile file;

      if (getEditorInput() instanceof IFileEditorInput) {
        file = ((IFileEditorInput) getEditorInput()).getFile();
        if (file.exists())
          saveContents(file);
        else
			doSaveAs(MessageFormat.format
				("The original file ''{0}'' has been deleted.", 
				new Object[] {file.getName()}));
      } else {
        doSaveAs();
      }
    } catch (CoreException e) {
      monitor.setCanceled(true);
      MessageDialog.openError(
        null,
        "Unable to Save Changes",
        e.getLocalizedMessage());
      return;
    }
  }

  /**
   * @see org.eclipse.ui.IEditorPart#doSaveAs()
   */
  public void doSaveAs() {
	doSaveAs(null);
  }
//Edu-Sol : 10b Save Editor content - new file
  private void doSaveAs(String message) {
    try {
      IFile file = createNewFile(message);
      if (file != null) {
        saveContents(file);
        setTitle(file.getName());
        setInput(new FileEditorInput(file));
        firePropertyChange(PROP_INPUT);
      }
    } catch (CoreException e) {
      MessageDialog.openError(
        null,
        "Unable to Save Changes",
        e.getLocalizedMessage());
      return;
    }
  }  

  private IFile createNewFile(String message) throws CoreException {
    SaveAsDialog dialog = new SaveAsDialog(getEditorSite().getShell());
    dialog.setTitle("Save Mini-Spreadsheet As");        
    if (getEditorInput() instanceof FileEditorInput)
      dialog.setOriginalFile(
        ((FileEditorInput) getEditorInput()).getFile());
   
    dialog.create();
    dialog.setMessage("Save file to another location.");
    if (message != null)
    	dialog.setMessage(message, IMessageProvider.WARNING);
    	
    dialog.open();
    IPath path = dialog.getResult();

    if (path == null) {
      return null;
    } else {
      String ext = path.getFileExtension();
      if (ext == null || !ext.equalsIgnoreCase("mss")) {
        throw new CoreException(
          new Status(
            IStatus.ERROR,
            MiniSpreadsheetUIPlugin.getId(),
            0,
            "File extension must be 'mss'.",
            null));
      }

      IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
      if (!file.exists())
        file.create(new ByteArrayInputStream(new byte[] {}), false, null);

      return file;
    }
  }

  /**
   * @see org.eclipse.ui.IEditorPart#gotoMarker(IMarker)
   */
  public void gotoMarker(IMarker marker) {
    // Called when user double-clicks an error in the Task list.
    // The marker has all the location information you need.		
  }

  /**
   * @see org.eclipse.ui.IEditorPart#init(IEditorSite, IEditorInput)
   */
  // Edu-Sol : 05a Editor input defined
  public void init(IEditorSite site, IEditorInput editorInput)
    throws PartInitException {

    if (!(editorInput instanceof IStorageEditorInput))
      throw new PartInitException("Invalid Input");

    init(site, (IStorageEditorInput) editorInput);
  }
  
//Edu-Sol : 05b Listener for changes to Editor input defined
  protected void init(IEditorSite site, IStorageEditorInput editorInput)
    throws PartInitException {

    try {
      setContents(editorInput);
      setSite(site);
      setInput(editorInput);
      setTitle(editorInput.getName());
      ResourcesPlugin.getWorkspace().addResourceChangeListener(
        this,
        IResourceChangeEvent.POST_CHANGE);
    } catch (CoreException e) {
      throw new PartInitException(e.getMessage());
    }
  }

//Edu-Sol : 05c Editor input mapped to local object
  protected void setContents(IStorageEditorInput sei)
    throws CoreException {
    InputStream is;

    is = sei.getStorage().getContents();

    miniSpreadsheet.removeMiniSpreadsheetListener(this);
    miniSpreadsheet = new MiniSpreadsheet(is);
    setIsDirty(false);
    miniSpreadsheet.addMiniSpreadsheetListener(this);
  }

  /**
   * @see com.ibm.lab.soln.msseditor.core.IMiniSpreadsheetListener#valueChanged(MiniSpreadsheet, int, int, String)
   */
//Edu-Sol : 09 React to changes
  public void valueChanged(
    MiniSpreadsheet miniSpreadsheet,
    int row,
    int column,
    String newValue) {
    setIsDirty(true);
  }

  /**
   * @see com.ibm.lab.soln.msseditor.core.IMiniSpreadsheetListener#rowsChanged(MiniSpreadsheet)
   */
  public void rowsChanged(MiniSpreadsheet miniSpreadsheet) {
    setIsDirty(true);
  }

  private void saveContents(IFile file) throws CoreException {
    ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
    try {
      miniSpreadsheet.save(file);
      setIsDirty(false);
    } finally {
      ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
    }
  }

  /**
   * @see org.eclipse.ui.IEditorPart#isDirty()
   */
  public boolean isDirty() {
    return isDirty;
  }

  protected void setIsDirty(boolean isDirty) {
    this.isDirty = isDirty;
    firePropertyChange(PROP_DIRTY);
  }

  /**
   * @see org.eclipse.ui.IEditorPart#isSaveAsAllowed()
   */
  public boolean isSaveAsAllowed() {
    return true;
  }

  /**
   * Set alignment for all the columns.
   * 
   * @param alignment SWT.LEFT, SWT.RIGHT, SWT.CENTER.
   */
  public void setAlignment(int alignment) {
    TableColumn[] tcs = table.getColumns();
    for (int i = 0; i < tcs.length; i++) {
      tcs[i].setAlignment(alignment);
    }
    this.defaultAlignment = alignment;
    table.redraw();
  }

  public int getDefaultAlignment() {
    return defaultAlignment;
  }

  /**
   * Returns the editor's model, the miniSpreadsheet.
   * 
   * @return MiniSpreadsheet
   */
  public MiniSpreadsheet getMiniSpreadsheet() {
    return miniSpreadsheet;
  }

  /**
   * This editor is a resource change listener in order to detect "special" situations.  Specifically:
   * 
   * <ul>
   * <li>Resource is deleted while editor is open (action: mark as dirty, permit only "Save As...")
   * <li>Resource is replaced by local history, modified by another means (view), 
   * or modified outside Eclipse and then user selects "Refresh" (action: update contents of editor)
   * </ul>
   * 
   * This editor supports both file-based and stream-based inputs. 
   * Note that the editor input can become file-based if the user
   * chooses "Save As...".
   * 
   * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(IResourceChangeEvent)
   */
//Edu-Sol : 11a Synch Editor content and model (source)
	public void resourceChanged(IResourceChangeEvent event) {
	
		// If the editor input is not a file, no point in worrying about resource changes.
		if (!(getEditorInput() instanceof FileEditorInput))
			return;
	
		if (event.getType() == IResourceChangeEvent.POST_CHANGE) {
			final IFile file = ((FileEditorInput) getEditorInput()).getFile();
			IResourceDelta delta = event.getDelta().findMember(file.getFullPath());
	
			if (delta != null) {
				if (delta.getKind() == IResourceDelta.REMOVED) {
	
					// Editor's underlying resource was deleted. Mark editor
					// as dirty and only allow "Save As..." (see doSave method
					// for more details).
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							setIsDirty(true);
						}
					});
				}
	
				if (delta.getKind() == IResourceDelta.CHANGED
					&& (delta.getFlags()
						& (IResourceDelta.CONTENT | IResourceDelta.REPLACED))
						!= 0) {
	
					// Editor's underlying resource has changed, perhaps by update in 
					// local history, refresh, etc. Note that this update cannot be
					// because of a change initiated by the editor, since the
					// editor removes its RCL during updates (see saveContents
					// for more details).
					Display.getDefault().syncExec(new Runnable() {
						public void run() {
							try {
								miniSpreadsheet.load(file.getContents());
								setIsDirty(false);
							} catch (CoreException e) {
								setIsDirty(true);
							}
						}
					});
				}
			}
		}
	}

  /**
   * @see org.eclipse.ui.IWorkbenchPart#dispose()
   */
  public void dispose() {
    super.dispose();

    ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
  }

}