package com.ibm.lab.soln.resources.nature_builder;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;

import java.lang.reflect.InvocationTargetException;

import com.ibm.lab.soln.resources.EDUResourcesPlugin;
import com.ibm.lab.soln.resources.IResourceIDs;
/**
 * Wizard to create a new project with the nature <code>CustomNature</code>.
 * 
 * Processing in this wizard is modeled after the platform new project wizard.
 * 
 * @see org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard
 */
public class NewProjectWizard extends Wizard implements INewWizard {

  // Reuse of the new project page provided by the platform UI dialogs 
  private WizardNewProjectCreationPage mainPage;
  // cache of newly-created project
  private IProject newProject;
  // switch to control write of trace data
  private boolean traceEnabled = false;

  /**
   * Add the one page to the wizard, the reused page 
   * <code>WizardNewProjectCreationPage</code>. This page provides
   * basic project name validation and allows for 
   * 
   * @see org.eclipse.jface.wizard.Wizard#addPages()
   */
  public void addPages() {
    mainPage = new WizardNewProjectCreationPage("MyProjNewPage");
    mainPage.setDescription(
      "Create a new project with CustomNature."
        + "\n CustomNature adds ReadmeBuilder.");
    mainPage.setTitle("New Custom Project");
    addPage(mainPage);
  }
  /**
   * Returns the newly created project.
   *
   * @return the created project, or <code>null</code>
   *   if project not created
   */
  public IProject getNewProject() {
    return newProject;
  }
  /**
   * Initializes this creation wizard using the passed workbench and
   * object selection.
   * <p>
   * This method is called after the no argument constructor and
   * before other methods are called.
   * </p>
   *
   * @param workbench the current workbench
   * @param selection the current object selection
   * 
   * @see org.eclipse.ui.IWorkbenchWizard#init(IWorkbench, IStructuredSelection)
   */
  public void init(IWorkbench workbench, IStructuredSelection selection) {

  }
  /**
   * Creates a project with a <code>CustomNature</code> association.
   * 
   * @return <code>true</code> to indicate the finish request
   *   was accepted, and <code>false</code> to indicate
   *   that the finish request was refused
   * 
   * @see org.eclipse.jface.wizard.IWizard#performFinish()
   */
  public boolean performFinish() {
    createNewProject();
    return true;
  }
  /**
   * Creates a new project resource with the name selected in the wizard page.
   * Project creation is wrapped in a <code>WorkspaceModifyOperation</code>.
   * <p>
   *
   * @see org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard#createNewProject()
   * @see org.eclipse.ui.actions.WorkspaceModifyOperation
   * 
   * @return the created project resource, or <code>null</code> if the project
   *    was not created
   */
  public IProject createNewProject() {
    if (newProject != null)
      return newProject;

    // get a project handle
    final IProject newProjectHandle = mainPage.getProjectHandle();

    // get a project descriptor
    IPath defaultPath = Platform.getLocation();
    IPath newPath = mainPage.getLocationPath();
    if (defaultPath.equals(newPath))
      newPath = null;
    IWorkspace workspace = EDUResourcesPlugin.getWorkspace();
    final IProjectDescription description =
      workspace.newProjectDescription(newProjectHandle.getName());
    description.setLocation(newPath);

    // create the new project operation
    WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
      protected void execute(IProgressMonitor monitor)
        throws CoreException {
        createProject(description, newProjectHandle, monitor);
        addCustomNature(newProjectHandle);
      }
    };

    // run the new project creation operation
    try {
      getContainer().run(false, true, op);
    } catch (InterruptedException e) {
      return null;
    } catch (InvocationTargetException e) {
      // ie.- one of the steps resulted in a core exception
      resultError(
        "Create Project with CustomNature Request",
        "Project creation failed");
      e.printStackTrace();

      return null;
    }

    newProject = newProjectHandle;

    return newProject;
  }
  /**
   * Creates a project resource given the project handle and description.
   *
   * @param description the project description to create a project resource for
   * @param projectHandle the project handle to create a project resource for
   * @param monitor the progress monitor to show visual progress with
   *
   * @exception CoreException if the operation fails
   * @exception OperationCanceledException if the operation is canceled
   */
  public void createProject(
    IProjectDescription description,
    IProject projectHandle,
    IProgressMonitor monitor)
    throws CoreException, OperationCanceledException {
    try {
      monitor.beginTask("", 2000);

      projectHandle.create(
        description,
        new SubProgressMonitor(monitor, 1000));

      if (monitor.isCanceled())
        throw new OperationCanceledException();

      projectHandle.open(new SubProgressMonitor(monitor, 1000));

    } finally {
      monitor.done();
    }
  }
  /**
   * Add the nature to the project.
   */
  public void addCustomNature(IProject project) throws CoreException {
    try {
      IProjectDescription description = project.getDescription();
      String[] natures = description.getNatureIds();
      String[] newNatures = new String[natures.length + 1];
      System.arraycopy(natures, 0, newNatures, 0, natures.length);
      newNatures[natures.length] = IResourceIDs.NATURE_ID;
      description.setNatureIds(newNatures);
      project.setDescription(description, null);
      resultInformation(
        "Create Project with CustomNature Request",
        "CustomNature added to the " + project.getName() + " project.");
    } catch (CoreException e) {
      // ie.- one of the steps resulted in a core exception
      resultError(
        "Create Project with CustomNature Request",
        "Adding CustomNature to project " + project.getName() + " failed");
      e.printStackTrace();
    }

  }
  /**
   * Write trace statements.  
   * System.out.println with prefix tagging used for simplicity.
   */
  void traceMsg(String msg) {
    if (traceEnabled)
      System.out.println(msg);
  }
  /**
  * Used to show action results.
  * 
  * @see org.eclipse.jface.dialogs.MessageDialog
  */
  protected void resultInformation(String title, String msg) {
    // Confirm Result

    if (traceEnabled)
      // trace only to console
      System.out.println(title + msg);
    else
      // user interaction response
      MessageDialog.openInformation(getShell(), title, msg);

  }

  /**
   * Used to show action results.
   * 
   * @see org.eclipse.jface.dialogs.MessageDialog
   */
  protected void resultError(String title, String msg) {
    // Indicate Error
    if (traceEnabled)
      // trace only to console
      System.out.println(title + msg);
    else
      // user interaction response
      MessageDialog.openError(getShell(), title, msg);

  }

}