package com.ibm.lab.soln.resources.nature_builder;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.runtime.CoreException;

import com.ibm.lab.soln.resources.ActionResourceAction;
import com.ibm.lab.soln.resources.IResourceIDs;

/**
 * The <code>ActionRemoveBuilderFromProject</code> implements an action 
 * to remove a builder from an existing project. 
 * <p>
 * This demonstrates the API for the project and project description 
 * workspace resources and how you might implement this logic in your tool.
 * </p>
 * @see org.eclipse.core.resources.IProject
 * @see org.eclipse.core.resources.IProjectDescription
 * @see org.eclipse.core.resources.ICommand
 */

public class ActionRemoveBuilderFromProject extends ActionResourceAction {

  /**
   * ActionRemoveBuilderFromProject default constructor.
   */
  public ActionRemoveBuilderFromProject() {
    super();
  }

  /**
   * Uses saved reference to selected project to call custom logic that
   * removes the builder from the current project.
   * 
   * @see org.eclipse.ui.IActionDelegate#run(IAction)
   * 
   * @param action action proxy that handles the presentation
   * portion of the plugin action
   */
  public void run(org.eclipse.jface.action.IAction action) {
    // Get project selected in Navigator
    // Extenstion definition filters to just single IProject selection
    IProject project = (IProject) this.selection.iterator().next();

    try {
      this.removeFromBuildSpec(project);

    } catch (CoreException e) {
      e.printStackTrace();

      // Problem with Builder Add
      resultError(
        "Soln: Resource - Remove ReadmeBuilder Request",
        "Error removing the ReadmeBuilder from the project.");
    }

  }

  /**
   * Adds a builder to a project description if it does not already exist.
   *
   * @param project  selected project resource is input for processing
   */
  public void removeFromBuildSpec(IProject project) throws CoreException {

    IProjectDescription desp = project.getDescription();
    ICommand[] commands = desp.getBuildSpec();
    boolean found = false;

    for (int i = 0; i < commands.length; ++i) {
      if (commands[i].getBuilderName().equals(IResourceIDs.BUILDER_ID)) {
        found = true;
        break;
      }
    }

    if (found) { //remove builder to project

      ICommand[] newCommands = new ICommand[commands.length - 1];

      // Add it before other builders.

      System.arraycopy(commands, 1, newCommands, 0, commands.length - 1);

      desp.setBuildSpec(newCommands);
      project.setDescription(desp, null);
      // Confirm Builder Remove
      resultInformation(
        "Soln: Resource - Remove ReadmeBuilder Request",
        "ReadmeBuilder was removed.");
    }
  }
}
