package com.ibm.lab.soln.resources.nature_builder;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */
 
import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.Platform;

import com.ibm.lab.soln.resources.ActionResourceAction;
import com.ibm.lab.soln.resources.IResourceIDs;

/**
 * Implements an action to add the <code>ReadmeBuilder</code> 
 * directly to an existing project. 
 * <p>
 * This demonstrates the API for the project and project description 
 * workspace resources and how you might implement this logic in your tool.
 * </p>
 * @see org.eclipse.core.resources.IProject
 * @see org.eclipse.core.resources.IProjectDescription
 * @see org.eclipse.core.resources.ICommand
 */

public class ActionAddBuilderToProject
  extends ActionResourceAction {

  /**
   * ActionAddBuilderToProject default constructor.
   */
  public ActionAddBuilderToProject() {
    super();
  }

  /**
   * Uses saved reference to selected project to call custom logic that
   * adds the builder to the current project.
   * 
   * @see org.eclipse.ui.IActionDelegate#run(IAction)
   * 
   * @param action action proxy that handles the presentation
   * portion of the plugin action
   */
  public void run(org.eclipse.jface.action.IAction action) {
    // Get project selected in Navigator
    // Extenstion definition filters to just single IProject selection
    IProject project =
      (IProject) this.selection.iterator().next();

    try {
      if (hasNature(IResourceIDs.BUILDER_ID))
        if (resultQuestion("Soln: Resource - Add ReadmeBuilder Request",
          "ReadmeBuilder spec says hasNature=\"true\", may not be able to add to project. Try anyway?"))
          this.addToBuildSpec(project);

    } catch (CoreException e) {
      e.printStackTrace();

      // Problem with Builder Add
      resultError(
        "Soln: Resource - Add ReadmeBuilder Request",
        "Error adding the ReadmeBuilder to the project.");

    }
  }

  /**
   * Adds the ReadmeBuilder to the project description for the selected project
   * if it does not already exist.
   *
   * @param project  selected project resource
   */

  public void addToBuildSpec(IProject project)
    throws CoreException {

    // get project description and then the associated build commands 
    IProjectDescription desc = project.getDescription();
    ICommand[] commands = desc.getBuildSpec();

    boolean found = hasBuilder(commands);

    // add builder if not already in project
    if (!found) {
      ICommand command = desc.newCommand();
      command.setBuilderName(IResourceIDs.BUILDER_ID);
      // create map with arguments specific to builder in project here 
      // command.setArguments(Map args);
      ICommand[] newCommands =
        new ICommand[commands.length + 1];

      // Add it before other builders. 
      System.arraycopy(
        commands,
        0,
        newCommands,
        1,
        commands.length);
      newCommands[0] = command;
      desc.setBuildSpec(newCommands);
      project.setDescription(desc, null);

      // Confirm Builder Add
      if (hasBuilder(project
        .getDescription()
        .getBuildSpec())) {

        resultInformation(
          "Soln: Resource - Add ReadmeBuilder Request",
          "ReadmeBuilder was added.");
        //        break;

      } else {
        // Problem with Builder Add
        resultError(
          "Soln: Resource - Add ReadmeBuilder Request",
          "Builder not added, Nature that references builder may exist.");

      }

    } else {

      // Builder already exists in project
      resultError(
        "Soln: Resource - Add ReadmeBuilder Request",
        "ReadmeBuilder already exists in the project.");
    }
  }

  /**
   * Checks project buildspec to determine if builder already associated.
   * 
   * @param commands - Project build spec
   * @return boolean
   */
 public boolean hasBuilder(ICommand[] commands) {

    boolean found = false;
    for (int i = 0; i < commands.length; ++i) {
      if (commands[i]
        .getBuilderName()
        .equals(IResourceIDs.BUILDER_ID)) {
        found = true;
        break;
      }
    }
    return found;
  }

  /**
   * Checks builder definition to determine if builder may require a nature.
   * Method source copied from <code>BuildManager.instantiateBuilder()</code> method.
   * 
   * @param commands - Project build spec
   * @return boolean
   */
 public boolean hasNature(String builderName) {

    IExtension extension =
      Platform.getPluginRegistry().getExtension(
        ResourcesPlugin.PI_RESOURCES,
        ResourcesPlugin.PT_BUILDERS,
        builderName);
    if (extension == null)
      return false;
    IConfigurationElement[] configs =
      extension.getConfigurationElements();
    if (configs.length == 0)
      return false;
    String hasNature = configs[0].getAttribute("hasNature");
    String natureId = null;
    if (hasNature != null
      && hasNature.equalsIgnoreCase(Boolean.TRUE.toString())) {
      //		//find the nature that owns this builder
      //		String builderId = extension.getUniqueIdentifier();
      //		natureId = EDUResourcesPlugin.getWorkspace().getNatureManager().findNatureForBuilder(builderId);
      //		if (natureId == null)
      return true;
    }
    return false;
  }

}
