package com.ibm.lab.soln.resources.markers;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */
 
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;

/**
 * Resource change listener that is used to track file modifications so 
 * that markers can be created to support a rapid re-open of these files.
 * <p>
 * This class is also the resource delta visitor that is used to process
 * the resource delta passed as part of the resource change event.
 * </p>
 */
public class RecentEditsRCL
  implements IResourceChangeListener, IResourceDeltaVisitor {

  RecentEdits redits = null;
  /**
   * Constructor for RecentEditsRCL which is used to instantiate the 
   * <code>RecentEdits</code> class which is passed to the 
   * <code>RecentEditsResourceDelta</code> visitor for further processing.
   */
  public RecentEditsRCL() {
    super();
    redits = new RecentEdits();
  }

  /**
   * Determine if the resource change event is a <code>POST_AUTO_BUILD</code> 
   * event, which is required as resources will be created and the workspace
   * is not locked during a <code>POST_AUTO_BUILD</code> event.
   * 
   * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(IResourceChangeEvent)
   */
  public void resourceChanged(IResourceChangeEvent event) {

    // only expect IResourceChangeEvent.POST_AUTO_BUILD - but checking anyway
    if (event.getType()
      == IResourceChangeEvent.POST_AUTO_BUILD) {
      try {
        event.getDelta().accept(this);
      } catch (CoreException e) {
        // See if the exception has anything to say...
        e.printStackTrace();
      }
    }
  }

  /**
    * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(IResourceDelta)
    */
  public boolean visit(IResourceDelta delta)
    throws CoreException {
    IResource res = delta.getResource();
    // Process resource if a file that was changed
    if ((res.getType() == IResource.FILE)) {
      if ((delta.getKind() == IResourceDelta.CHANGED)
        && (delta.getFlags() == IResourceDelta.CONTENT)) {
        // ask recent edits to add a new marker
        redits.addReditMarker(res);
      }
      // Process resource if it is a file and a marker was deleted
      if ((res.getType() == IResource.FILE)) {
        if ((delta.getKind() == IResourceDelta.CHANGED)
          && ((delta.getFlags() & IResourceDelta.MARKERS)
            != 0)) {
          // ask recent edits to remove from list 
          redits.checkReditMarker(res, delta);
        }
      }
    }
    return true; // visit the kids too!
  }
}
