package com.ibm.lab.soln.resources;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.ISaveContext;
import org.eclipse.core.resources.ISaveParticipant;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;

import java.io.File;
/**
 * Resource programming demonstration of the structure of a 
 * participant in Workspace save events.  Tied to logic in the plug-in's 
 * <code>startup()</code> method that adds a save participant to the workspace.
 * <p>
 * Tools that need to save project-specific information should consider using 
 * a save participant so that you can be notified when new projects are created 
 * and when changes that trigger a save event have occurred.  This allows your 
 * tool logic to save any information that may be required to support the 
 * operation of your tool.
 * </p>
 * @see org.eclipse.core.resources.ISaveParticipant
 * @see org.eclipse.core.resources.ISaveContext
 */
public class WorkspaceSaveParticipant implements ISaveParticipant {
  // switch to control write of trace data
  private boolean traceEnabled = true;
  // Value used to prefix trace messages
  String wspMode = null;
  // Value used to indicate how many times the save participant 
  // has been invoked. Used as part of the prefix in trace messages.
  int wspCount = 0;

  /**
   * Custom constructor to set an alternate prefix mode that will be 
   * used for console message tagging.
   */
  public WorkspaceSaveParticipant() {
    super();
    wspMode = "WkSavePart -> Save Count:";
    traceMsg(" *----------> MyWorkSpaceSaveParticipant created.", null);
  }
  
  /**
   * Tells this participant that the workspace save operation is now
   * complete and it is free to go about its normal business.
   *
   * @see org.eclipse.core.resources.ISaveParticipant#doneSaving
   * @param context the save context object
   */
  public void doneSaving(ISaveContext context) {
    traceMsg(" *----------> doneSaving ", context);
  }
  
  /**
   * Tells this participant that the workspace is about to be
   * saved. Local count of how many times this process is called is incremented.
   * 
   * @see org.eclipse.core.resources.ISaveParticipant#prepareToSave
   *
   * @param context the save context object
   * @exception CoreException if this method fails to snapshot
   *   the state of this workspace 
   */
  public void prepareToSave(ISaveContext context) throws CoreException {

    wspCount++;
    traceMsg(" *----------> prepareToSave ", context);
  }
  
  /**
   * Tells this participant to rollback its important state.
   *
   * @see org.eclipse.core.resources.ISaveParticipant#rollback
   * 
   * @param context the save context object
   * @see ISaveContext#getPreviousSaveNumber
   */
  public void rollback(ISaveContext context) {
    traceMsg(" *----------> rollback ", context);
  }
  
  /**
   * Tells this participant that the workspace is about to be
   * saved. In preparation, the participant is expected to suspend
   */
  public void saveFile(File f) throws CoreException {
    try {
      f.createNewFile();
    } catch (java.io.IOException e) {

    }
  }

  /**
   * Tells this participant that the workspace is about to be
   * saved. 
   * 
   * @see org.eclipse.core.resources.ISaveParticipant#saving
   *
   * @param context the save context object
   * @exception CoreException if this method fails to snapshot
   *   the state of this workspace 
   */
  public void saving(ISaveContext context) throws CoreException {
    EDUResourcesPlugin plugin = EDUResourcesPlugin.getDefault();
    int saveNumber = context.getSaveNumber();
    String saveFileName = "save-" + Integer.toString(saveNumber);
    traceMsg(
      " *----------> saving - Save File Name: " + saveFileName,
      context);

    File f = plugin.getStateLocation().append(saveFileName).toFile();
    // here is where you might save something you need to find later
    //	plugin.writeImportantState(f);
    saveFile(f);

    context.map(new Path("save"), new Path(saveFileName));
    context.needSaveNumber();
    context.needDelta(); // optional

  }

  /**
   * Write trace statements.  
   * System.out.println with prefix tagging used for simplicity.
   */
  void traceMsg(String msg, ISaveContext context) {
    if (traceEnabled)
      System.out.println(wspMode + wspCount + msg);
    if (context != null)
      switch (context.getKind()) {
        case ISaveContext.FULL_SAVE :
          System.out.println("ISaveContext.FULL_SAVE");
          break;
        case ISaveContext.SNAPSHOT :
          System.out.println("ISaveContext.SNAPSHOT");
          break;
        case ISaveContext.PROJECT_SAVE :
          System.out.println("ISaveContext.PROJECT_SAVE");
          break;

        default :
          break;
      }
  }

}