package com.ibm.lab.soln.resources;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.internal.events.ResourceDelta;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;

/**
 * Visitor for <code>IResourceDelta</code> to find content and information 
 * and print trace messages to show this during a testing cycle.
 * 
 * Use this as a visitor for a builder or a resource change listener to 
 * find out more about the content of the <code>IResourceDelta</code>.
 */
public class ResourceDeltaPrinter
	implements org.eclipse.core.resources.IResourceDeltaVisitor {
	// switch to control write of trace data
	private boolean traceEnabled = false;
	// Value used to prefix trace messages
	String rclMode = null;
	/**
	 * Default constructor which sets a default mode value.
	 * Mode is used to tag messages to the console to help visualize the 
	 * trace data generated.
	 */
	public ResourceDeltaPrinter() {
		super();
		rclMode = "\tDeltaPrinter:  ";
	}

	/**
	 * Alternate constructor to set an alternate prefix mode that will be 
	 * used for console message tagging.
	 */
	public ResourceDeltaPrinter(String mode) {
		this();
		//    rclMode = mode + " -> \tDeltaPrinter:  ";
		rclMode = mode + " -> ";
	}

	/** 
	 * Visits the given resource delta and prints trace statements to indicate
	 * content of the delta.  Follows markers when they exist in the delta.
	 * 
	 * @return <code>true</code> if the resource delta's children should
	 *      be visited; <code>false</code> if they should be skipped.
	 */

	public boolean visit(IResourceDelta delta) {
		IResource res = delta.getResource();

		ResourceDelta rdelta = (ResourceDelta) delta;
		//      traceMsg("debug :" + rdelta.toDebugString());

		switch (delta.getKind()) {
			case IResourceDelta.ADDED :
				traceMsg("\tResource " + res.getFullPath() + " was added.");
				break;
			case IResourceDelta.REMOVED :
				traceMsg("\tResource " + res.getFullPath() + " was removed.");
				break;
			case IResourceDelta.CHANGED :
				traceMsg("\tResource " + res.getFullPath() + " has changed.");
				break;
		}
		// Report flag detail
		testFlags(delta.getFlags());
		if ((delta.getKind()
			== IResourceDelta.CHANGED | delta.getKind()
			== IResourceDelta.ADDED)
			&& (delta.getFlags() & IResourceDelta.MARKERS) != 0) {
			try {
				// traceMsg("Resource " + res.getFullPath());
				//	traceMsg("\t> Marker changes:");
				processMarkerSet(res, delta);
			} catch (CoreException e) {
				e.printStackTrace();

			}

		}

		//        break;
		//    }
		return true; // visit the kids too!
	}

	/**
	 * Prints delta event kind messages to the console 
	 */
	public void testFlags(int eventFlag) {
		// was wrong in book text

		if ((eventFlag & IResourceDelta.OPEN) != 0)
			traceMsg("\tResourceDelta Kind: Opened");
			
		if ((eventFlag & IResourceDelta.CONTENT) != 0)
			traceMsg("\tResourceDelta Kind: Content Change");

		if ((eventFlag & IResourceDelta.REPLACED) != 0)
			traceMsg("\tResourceDelta Kind: Content Replaced");

		if ((eventFlag & IResourceDelta.REMOVED) != 0)
			traceMsg("\tResourceDelta Kind: Removed");

		if ((eventFlag & IResourceDelta.MARKERS) != 0)
			traceMsg("\tResourceDelta Kind: Marker Change");

	}

	/**
	 * Prints delta marker information messages to the console 
	 */
	public void processMarkerSet(IResource res, IResourceDelta delta)
		throws CoreException {

		IMarkerDelta[] markers = delta.getMarkerDeltas();

		for (int i = 0; i < markers.length; i++) {
			IMarkerDelta markerDelta = markers[i];
			int kind = markerDelta.getKind();
			traceMsg("\t\t Marker delta kind: " + getKind(kind));
			IMarker marker = markerDelta.getMarker();
			//      traceMsg("\t Marker itself: " + marker);
			if (marker.exists()) {
				traceMsg("\t\t Marker type: " + marker.getType());
				traceMsg("\t\t Marker content: " + marker.getAttributes().values());
			}
//			traceMsg("<--------------------");
		}

	}

	private String getKind(int kind) {
		switch (kind) {
			case IResourceDelta.ADDED :
				return "Added";

			case IResourceDelta.CHANGED :
				return "Changed";

			case IResourceDelta.REMOVED :
				return "Removed";

			default :
				return "Other";
		}
	}

	/**
	 * Write trace statements.  
	 * System.out.println with prefix tagging used for simplicity.
	 */
	void traceMsg(String msg) {
		if (IResourceIDs.TRACE_ENABLED | traceEnabled)
			System.out.println(rclMode + msg);
	}

}