package com.ibm.lab.msseditor.core;

/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;

/**
 * Represents a single row of a mini-spreadsheet.
 * 
 * @see MiniSpreadsheet
 */
public class MiniSpreadsheetRow implements IAdaptable {
	private String data[];
	private int rowIndex;
	private MiniSpreadsheet miniSpreadsheet;

	/**
	 * Create a new row.
	 */	
	public MiniSpreadsheetRow(
		MiniSpreadsheet mss,
		int columnCount,
		int rowIndex) {
		data = new String[columnCount];
		for (int i = 0; i < columnCount; i++) {
			data[i] = new String();
		}
		this.rowIndex = rowIndex;
		this.miniSpreadsheet = mss;
	}

	/**
	 * Get the value at [column] as a string.
	 */
	public String getString(int columnIndex) {
		if (columnIndex < 0 || columnIndex >= data.length)
			return "";
		else
			return data[columnIndex];
	}
	
	/**
	 * Get the value at [column] as an integer or zero if not
	 * a value integer format.
	 */
	public int getInt(int columnIndex) {
		if (columnIndex < 0 || columnIndex >= data.length)
			return 0;
		
		try {
			return Integer.parseInt(data[columnIndex]);
		}
		catch (NumberFormatException e) {
			return 0;
		}
	}
	
	/**
	 * Return <code>true</code> if the data at [column]
	 * is a valid integer.
	 */
	public boolean isInt(int columnIndex) {
		if (columnIndex < 0 || columnIndex >= data.length)
			return false;
		
		if (data[columnIndex].length() == 0)
			return false;
			
		try {
			Integer.parseInt(data[columnIndex]);
			return true;
		}
		catch (NumberFormatException e) {
			return false;
		}
	}	
	
	/**
	 * Return <code>true</code> if any data
	 * in this row is a valid integer.
	 */
	public boolean hasInts() {
		for (int i=0; i < data.length; i++)
			if (isInt(i))
				return true;
				
		return false;
	}		

	/**
	 * Set the value at [column] to string. 
	 * 
	 * <note>Since the
	 * persistent data is semi-colon delimited, semi-colons
	 * are translated to underscores.</note>
	 */
	public void setData(int columnIndex, String value) {
		if (columnIndex < 0 || columnIndex >= data.length)
			return;
			
		value.replace(';', '_');
		data[columnIndex] = value;
		miniSpreadsheet.fireCellChanged(rowIndex, columnIndex, value);
	}
	
	/**
	 * Set the value at [column] to integer.
	 */
	public void setData(int columnIndex, int value) {
		if (columnIndex < 0 || columnIndex >= data.length)
			return;
		
		data[columnIndex] = Integer.toString(value);
		miniSpreadsheet.fireCellChanged(rowIndex, columnIndex, data[columnIndex]);
	}	

	/**
	 * Return the number of columns.
	 */
	public int getColumnCount() {
		return data.length;
	}

	/**
	 * Return the row number of this instance in the mini-spreadsheet.
	 */
	public int getRowIndex() {
		return rowIndex;
	}
	
	/**
	 * Remove this instance from the mini-spreadsheet.
	 */
	public void remove() {
		miniSpreadsheet.removeRow(this);
	}

	void setRowIndex(int i) {
		rowIndex = i;
	}
	
	/**
	 * Allow instances of this class to be adapted to other interfaces.
	 * This is required to implement action filters.
	 * 
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(Class)
	 * @see com.ibm.lab.soln.msseditor.ui.actions.MiniSpreadsheetRowActionFilter
	 * @see com.ibm.lab.soln.msseditor.ui.MiniSpreadsheetUIPlugin#startup
	 */
	public Object getAdapter(Class adapter) {
       return Platform.getAdapterManager().getAdapter(this, adapter);
	}
}
