(let ((time-stamp "Time-stamp: <2004-03-16 12:23:40 wand>"))
  (eopl:printf "classes.scm: ~a~%"
    (substring time-stamp 13 29)))

;;; Revision History:

;;; Thu Mar 11 21:31:52 2004 redone from book files for simpler
;;; presentation. 

;;; extends implicit-store.scm 

;;; test with (run-all)

; uncomment for PLT
; (load "test-harness.scm")
; (load "test-suite.scm")


;;;;;;;;;;;;;;;; grammatical specification ;;;;;;;;;;;;;;;;

(define the-lexical-spec
  '((whitespace (whitespace) skip)
    (comment ("%" (arbno (not #\newline))) skip)
    (identifier
      (letter (arbno (or letter digit "_" "-" "?")))
      make-symbol)
    (number (digit (arbno digit)) make-number)))

(define the-grammar
  '((program ((arbno class-decl) expression) a-program)

    (expression (number) lit-exp)
    (expression (identifier) var-exp)
    (expression
      (primitive "(" (separated-list expression ",") ")")
      primapp-exp)
    (expression
      ("if" expression "then" expression "else" expression)
      if-exp)
   (expression
      ("let" (arbno  identifier "=" expression) "in" expression)
      let-exp)
    (expression
      ("proc" "(" (separated-list identifier ",") ")" expression)
      proc-exp)
    (expression
      ("(" expression (arbno expression) ")")
      app-exp)
    (expression                         
      ("letrec"
        (arbno identifier "(" (separated-list identifier ",") ")"
          "=" expression)
        "in" expression)
      letrec-exp)
    (expression ("set" identifier "=" expression) varassign-exp)
    (expression
      ("begin" expression (arbno ";" expression) "end")
      begin-exp)

    (primitive ("+")     add-prim)
    (primitive ("-")     subtract-prim)
    (primitive ("*")     mult-prim)
    (primitive ("add1")  incr-prim)
    (primitive ("sub1")  decr-prim)
    (primitive ("zero?") zero-test-prim)

    (primitive ("list") list-prim)
    (primitive ("cons") cons-prim)
    (primitive ("nil")  nil-prim)
    (primitive ("car")  car-prim)
    (primitive ("cdr")  cdr-prim)
    (primitive ("null?") null?-prim)

    ;; new productions for oop 

    (class-decl                         
      ("class" identifier 
        "extends" identifier                   
	 (arbno "field" identifier)
         (arbno method-decl)
         )
      a-class-decl)

    (method-decl
      ("method" identifier              ; method name
        "("  (separated-list identifier ",") ")" ; method formals
        expression                      ; method body
        )
      a-method-decl)

    (expression 
      ("new" identifier "(" (separated-list expression ",") ")")
      new-object-exp)

    (expression
      ("send" expression identifier
        "("  (separated-list expression ",") ")")
      method-app-exp)

    (expression                         ; self must be special because
                                        ; it's not settable.
      ("self")
      self-exp)

    (expression                                
      ("super" identifier    "("  (separated-list expression ",") ")")
      super-call-exp)

    ))


;;;;;;;;;;;;;;;; sllgen boilerplate ;;;;;;;;;;;;;;;;

(sllgen:make-define-datatypes the-lexical-spec the-grammar)

(define show-the-datatypes
  (lambda () (sllgen:list-define-datatypes the-lexical-spec the-grammar)))

(define scan&parse
  (sllgen:make-string-parser the-lexical-spec the-grammar))

(define just-scan
  (sllgen:make-string-scanner the-lexical-spec the-grammar))

;;;;;;;;;;;;;;;; interface to test harness ;;;;;;;;;;;;;;;;

(define run
  (lambda (string)
    (eval-program (scan&parse string))))

(define run-all
  (lambda ()
    (run-experiment run use-execution-outcome
      all-groups all-tests equal?)))

(define run-one
  (lambda (test-name)
    (run-test run test-name)))

(define all-groups '(lang3-5 lang3-6 lang3-7 oop))

;;;;;;;;;;;;;;;; the interpreter ;;;;;;;;;;;;;;;;

(define eval-program 
  (lambda (pgm)
    (cases program pgm
      (a-program (class-decls body)
        (initialize-store!)
        (elaborate-class-decls! class-decls) ; new for ch6
        (eval-expression body (init-env))))))

(define eval-expression
  (lambda (exp env)
    (cases expression exp
      (lit-exp (datum) datum)
     (var-exp (id) (deref (apply-env env id)))
      (primapp-exp (prim rands)
        (let ((args (eval-rands rands env)))
          (apply-primitive prim args)))
      (if-exp (test-exp true-exp false-exp)
        (if (true-value? (eval-expression test-exp env))
          (eval-expression true-exp env)
          (eval-expression false-exp env)))
      (let-exp (ids rands body)
	(let ((args (eval-rands rands env)))
          (eval-expression body (extend-env ids args env))))
      (proc-exp (ids body)
        (closure ids body env))
      (app-exp (rator rands)
        (let ((proc (eval-expression rator env))
              (args (eval-rands      rands env)))
          (if (procval? proc)
            (apply-procval proc args)
            (error 'eval-expression 
              "attempt to apply non-procedure ~s"
              proc))))
      (letrec-exp (proc-names idss bodies letrec-body)
        (eval-expression letrec-body
          (extend-env-recursively proc-names idss bodies env)))
      (varassign-exp (id rhs-exp)
        (setref!
          (apply-env env id)
          (eval-expression rhs-exp env))
        1)
      (begin-exp (exp1 exps)
        (let loop ((acc (eval-expression exp1 env))
                   (exps exps))
          (if (null? exps) acc
            (loop (eval-expression (car exps) env) (cdr exps)))))

      ;; new cases for oop:

      (new-object-exp (class-name rands)
        (let ((args (eval-rands rands env))
              (obj (new-object class-name)))
          (apply-method
            (find-method 'initialize class-name)
            obj args)
          obj))

      (method-app-exp (obj-exp method-name rands)
        (let ((args (eval-rands rands env))
              (obj (eval-expression obj-exp env)))
          (apply-method
            (find-method method-name (object->class-name obj))
            obj args)))

      (self-exp ()
        (deref (apply-env env '%self)))

      (super-call-exp (method-name rands)
        (let ((args (eval-rands rands env))
              (obj (deref (apply-env env '%self))))
          (apply-method
            (find-method method-name (deref (apply-env env '%super)))
            obj args)))

      )))
      

(define eval-rands
  (lambda (exps env)
    (map
      (lambda (exp) (eval-expression exp env))
      exps)))

(define apply-primitive
  (lambda (prim args)
    (cases primitive prim
      (add-prim  () (+ (car args) (cadr args)))
      (subtract-prim () (- (car args) (cadr args)))
      (mult-prim  () (* (car args) (cadr args)))
      (incr-prim  () (+ (car args) 1))
      (decr-prim  () (- (car args) 1))
      (zero-test-prim () (if (zero? (car args)) 1 0))
      (list-prim () args)               ;already a list
      (nil-prim () '())
      (car-prim () (car (car args)))
      (cdr-prim () (cdr (car args)))
      (cons-prim () (cons (car args) (cadr args)))
      (null?-prim () (if (null? (car args)) 1 0))
      )))

(define init-env 
  (lambda ()
    (extend-env
      '(i v x)
      '(1 5 10)
      (empty-env))))

;;;;;;;;;;;;;;;; booleans ;;;;;;;;;;;;;;;;

(define true-value?
  (lambda (x)
    (not (zero? x))))

;;;;;;;;;;;;;;;; type constructors ;;;;;;;;;;;;;;;;

(define vector-of			; a cheat: it doesn't test contents
  (lambda (pred)
    (lambda (x) (vector? x))))

(define optional                        
  (lambda (pred)                        ; either false or satisfies pred
    (lambda (x) (or (not x) (pred x)))))

;;;;;;;;;;;;;;;; objects ;;;;;;;;;;;;;;;;

(define-datatype object object? 
  (an-object
    (class-name symbol?)                ; could be class?, but then
                                        ; objects become unprintable.
    (fields (vector-of reference?))))   ; suitable for becoming a rib

(define new-object                  
  (lambda (class-name)
    (an-object
      class-name
      (list->vector
        (map
          (lambda (n) (newref 'uninitialized-field))
          (class-name->field-ids class-name))))))

;;;;;;;;;;;;;;;; methods ;;;;;;;;;;;;;;;;

(define-datatype method method?
  (a-method
    (method-formals (list-of symbol?))
    (body expression?)
    (field-ids (list-of symbol?))
    (super-name symbol?)))

(define apply-method
  (lambda (meth obj args)
    (cases method meth
      (a-method (method-formals body field-ids super-name)
        (eval-expression
          body
          (extend-env
            (cons '%self (cons '%super method-formals))
            (cons obj (cons super-name args))
            (extend-env-with-vector
              field-ids
              (object->fields obj)
              (empty-env))))))))

;;;;;;;;;;;;;;;; classes ;;;;;;;;;;;;;;;;

(define-datatype class class?
  (a-class
    (super-name   symbol?) 
    (field-ids    (list-of symbol?))
    (method-ids   (list-of symbol?))
    (methods      (list-of method?))))

(define find-method
  (lambda (method-name class-name)
    (cases class (lookup-class class-name)
      (a-class (super-name field-ids method-ids methods)
        (let ((p (list-find-position method-name method-ids)))
          (if p
            (list-ref methods p)
            (eopl:error 'find-method
              "Can't find method ~s in class ~s"
              method-name class-name)))))))

(define elaborate-class-decls!
  (lambda (class-decls)
    (initialize-class-table!)
    (for-each elaborate-class-decl! class-decls)))

(define elaborate-class-decl!
  (lambda (c-decl)
    (cases class-decl c-decl
      (a-class-decl (class-name super-name local-fields local-method-decls)
        (let ((field-ids
                (assemble-field-ids
                  local-fields
                  (class-name->field-ids super-name)))
              (method-ids
                (assemble-method-ids
                  local-method-decls
                  super-name)))
          (add-to-class-table!
            class-name
            (a-class super-name field-ids method-ids
              (assemble-methods
                method-ids field-ids super-name local-method-decls))))))))


;; like (append old-ids new-ids), but if an old-id is shadowed by a
;; new-id, it's replaced by '%shadowed.  Thank you, Dan!

(define assemble-field-ids
  (lambda (new-ids old-ids)
    (cond
      ((null? old-ids) new-ids)
      ((memq (car old-ids) new-ids)
       (cons '%shadowed (assemble-field-ids new-ids (cdr old-ids))))
      (else
        (cons (car old-ids) (assemble-field-ids new-ids (cdr old-ids)))))))

;; like (append super-method-ids local-method-ids), but local methods
;; that override are ignored, because they will go in the slot of the
;; overridden method.

(define assemble-method-ids
  (lambda (local-method-decls super-name)
    (let ((local-method-ids
            (map method-decl->method-name local-method-decls))
          (super-method-ids (class-name->method-ids super-name)))
      (append
        super-method-ids
        (difference local-method-ids super-method-ids)))))

(define assemble-methods
  (lambda (method-ids field-ids super-name local-method-decls)
    (let
      ((local-method-alist   ; first assemble a-list of local methods:
         (map
           (lambda (m-decl)
             (cases method-decl m-decl  ; destructure the method-decl
               (a-method-decl (method-name method-formals method-body)
                 (cons method-name      ; key for association list
                   (a-method            ; and build the method
                     method-formals method-body field-ids
                     super-name))))) 
           local-method-decls)))
      (map                              ; now build a list with the
                                        ; right method for each name
        (lambda (m-name)
          (let ((p (assq m-name local-method-alist)))
            (if p
              (cdr p)                   ; get local (overriding)
                                        ; method first
              (find-method m-name super-name))))
        method-ids))))

;;;;;;;;;;;;;;;; class environments ;;;;;;;;;;;;;;;;

;;; class-env = (list-of (pair class-name class))

(define the-class-table 'uninitialized-class-table)

(define initialize-class-table!
  (lambda ()
    (set! the-class-table '())))

(define add-to-class-table!
  (lambda (class-name class)
    (set! the-class-table
      (cons
        (cons class-name class)
        the-class-table))))

;; sexy version using Scheme "=>" idiom
;; (define lookup-class
;;   (lambda (name)
;;     (cond
;;       ((assq name the-class-table) => cdr)
;;       (else #f))))

;; not so sexy version with assq

(define lookup-class
  (lambda (name)
    (let ((p (assq name the-class-table)))
      (if p (cdr p) #f))))

;; even less sexy version with assq replaced by explicit lookup:

;; (define lookup-class
;;   (lambda (name)
;;     (letrec ((loop (lambda (classes)
;;                      (cond
;;                        ((null? classes) #f)
;;                        ((eqv? (caar classes) name) (cdar classes))
;;                        (else (loop (cdr classes)))))))
;;       (loop the-class-table))))

;;;;;;;;;;;;;;;; selectors ;;;;;;;;;;;;;;;;

(define object->class-name
  (lambda (obj)
    (cases object obj
      (an-object (class-name fields)
        class-name))))

(define object->fields
  (lambda (obj)
    (cases object obj
      (an-object (class-decl fields)
        fields))))

(define class-name->field-ids
  (lambda (class-name)
    (if (eqv? class-name 'object) '()
      (class->field-ids (lookup-class class-name)))))

(define class-name->method-ids
  (lambda (class-name)
    (if (eqv? class-name 'object) '()
      (class->method-ids (lookup-class class-name)))))

(define class-name->methods
  (lambda (class-name)
    (if (eqv? class-name 'object) '()
      (class->methods (lookup-class class-name)))))

(define class->field-ids
  (lambda (c)
    (cases class c
      (a-class (super-name field-ids method-ids methods)
	field-ids))))

(define class->method-ids
  (lambda (c)
    (cases class c
      (a-class (super-name field-ids method-ids methods)
	method-ids))))

(define class->methods
  (lambda (c)
    (cases class c
      (a-class (super-name field-ids method-ids methods)
	methods))))

(define method-decl->method-name
  (lambda (md)
    (cases method-decl md
      (a-method-decl (method-name ids body) method-name))))
	
;;;; only change below here is extend-env-with-vector

;;;;;;;;;;;;;;;; procedures ;;;;;;;;;;;;;;;;

(define-datatype procval procval?
  (closure 
    (ids (list-of symbol?)) 
    (body expression?)
    (env environment?)))

(define apply-procval
  (lambda (proc args)
    (cases procval proc
      (closure (ids body env)
        (eval-expression body (extend-env ids args env))))))

;;;;;;;;;;;;;;;; references and the store ;;;;;;;;;;;;;;;;

;;; world's dumbest model of the store:  the store is a list and a
;;; reference is number which denotes a position in the list.

(define empty-store
  (lambda () '()))

(define initialize-store!
  (lambda ()
    (set! the-store (empty-store))))

(define reference? integer?)

(define newref
  (lambda (val)
    (let ((next-ref (length the-store)))
      (set! the-store
        (append the-store (list val)))  ; YUCK!
      next-ref))) 

(define deref 
  (lambda (loc) (list-ref the-store loc)))

(define setref!                          ; YUCK!
  (lambda (loc0 val)
    (set! the-store
      (let recur ((store the-store) (loc loc0))
        (cond
          ((null? store)
           (eopl:error 'setref
             "illegal reference ~s in store ~s"
             loc0 the-store))
          ((zero? loc)
           (cons val (cdr store)))
          (else
            (cons
              (car store)
              (recur (cdr store) (- loc 1)))))))))

;;;;;;;;;;;;;;;; environments ;;;;;;;;;;;;;;;;

(define-datatype environment environment?
  (empty-env-record)
  (extended-env-record
    (syms (list-of symbol?))
    (vec (vector-of reference?))           
    (env environment?))
  )

(define empty-env
  (lambda ()
    (empty-env-record)))

(define extend-env
  (lambda (syms vals env)
    (extended-env-record syms
      (list->vector (map newref vals))                 
      env)))

(define apply-env
  (lambda (env sym)
    (cases environment env
      (empty-env-record ()
        (eopl:error 'apply-env-ref "No binding for ~s" sym))
      (extended-env-record (syms vals env)
        (let ((pos (rib-find-position sym syms)))
          (if (number? pos)
            (vector-ref vals pos)
            (apply-env env sym)))))))

(define extend-env-recursively
  (lambda (proc-names idss bodies old-env)
    (let ((len (length proc-names)))
      (let ((vec (make-vector len)))
        (let ((env (extended-env-record proc-names vec old-env)))
          (for-each
            (lambda (pos ids body)
              (vector-set! vec pos
                (newref (closure ids body env)))) ; change
            (iota len) idss bodies)
          env)))))


(define extend-env-with-vector
  (lambda (syms vec env)
    (extended-env-record syms vec env)))

(define rib-find-position 
  (lambda (sym los)
    (list-find-position sym los)))

(define list-find-position
  (lambda (sym los)
    (list-index (lambda (sym1) (eqv? sym1 sym)) los)))

(define list-index
  (lambda (pred ls)
    (cond
      ((null? ls) #f)
      ((pred (car ls)) 0)
      (else (let ((list-index-r (list-index pred (cdr ls))))
              (if (number? list-index-r)
                (+ list-index-r 1)
                #f))))))

(define iota
  (lambda (end)
    (let loop ((next 0))
      (if (>= next end) '()
        (cons next (loop (+ 1 next)))))))

(define difference
  (lambda (set1 set2)
    (cond
      ((null? set1) '())
      ((memv (car set1) set2)
       (difference (cdr set1) set2))
      (else (cons (car set1) (difference (cdr set1) set2))))))
