;;; interpreter for OOP language with static classes

(define basis:time-stamp "Time-stamp: <2000-03-07 09:55:47 dfried>")

;;; Revision History:

;;; Thu Nov 11 15:17:31 1999 this splits off common pieces into basis.scm.

;;; started Tue Jul 13 16:36:03 1999.  Base language is that of 
;;; section 3.7, but with different environment structures.  

(let ((version "99-12")
      (date  (substring basis:time-stamp 13 30)))
  (printf "oop-basis.scm - version ~a ~a~%" version date))

;;;;;;;;;;;;;;;; grammatical specification ;;;;;;;;;;;;;;;;

(define the-lexer
  '((whitespace (whitespace) skip)
    (comment ("%" (arbno (not #\newline))) skip)
    (identifier
      (letter (arbno (or letter digit "_" "-" "?")))
      make-symbol)
    (number (digit (arbno digit)) make-number)))

(define grammar-with-inheritance
  '((program ((arbno class-decl) expression) a-program)

    (expression (number) lit-exp)
    (expression (identifier) var-exp)   ; book calls this <id>
    (expression
      (primitive "(" (separated-list expression ",") ")")
      primapp-exp)
    (expression
      ("if" expression "then" expression "else" expression)
      if-exp)
   (expression
      ("let" (arbno  identifier "=" expression) "in" expression)
      let-exp)
    (expression
      ("proc" "(" (separated-list identifier ",") ")" expression)
      proc-exp)
    (expression
      ("(" expression (arbno expression) ")")
      app-exp)
    (expression                         ; 3-6
      ("letrec"
        (arbno identifier "(" (separated-list identifier ",") ")"
          "=" expression)
        "in" expression)
      letrec-exp)
    (expression ("set" identifier "=" expression) varassign-exp)
    (expression
      ("begin" expression (arbno ";" expression) "end")
      begin-exp)

    (primitive ("+")     add-prim)
    (primitive ("-")     subtract-prim)
    (primitive ("*")     mult-prim)
    (primitive ("add1")  incr-prim)
    (primitive ("sub1")  decr-prim)
    (primitive ("zero?") zero-test-prim)
    (primitive ("list") list-prim)
    (primitive ("cons") cons-prim)
    (primitive ("nil")  nil-prim)
    (primitive ("car")  car-prim)
    (primitive ("cdr")  cdr-prim)
    (primitive ("null?") null?-prim)
    (let-decl
      (identifier "=" expression)
      a-let-decl)
    (proc-decl 
      (identifier "(" (separated-list identifier ",") ")" "=" expression)
      a-proc-decl)

;;;;;;;;;;;;;;;; new productions for oop ;;;;;;;;;;;;;;;;

    (class-decl                         
      ("class" identifier 
        "extends" identifier                   
	 (arbno "field" identifier)
         (arbno method-decl)
         )
      a-class-decl)

;     (field-decl                         
;       ("field" identifier)
;       a-field-decl)

    (method-decl
      ("method" identifier 
        "("  (separated-list identifier ",") ")" ; method ids
        expression 
        )
      a-method-decl)

    (expression 
      ("new" identifier "(" (separated-list expression ",") ")")
      new-object-exp)

    (expression
      ("send" expression identifier
        "("  (separated-list expression ",") ")")
      method-app-exp)

    (expression                                
      ("super" identifier    "("  (separated-list expression ",") ")")
      super-call-exp)

;;;;;;;;;;;;;;;; end new productions for oop ;;;;;;;;;;;;;;;;

    ))

(define show-the-datatypes
  (lambda () (sllgen:show-define-datatypes the-lexer the-grammar)))

(define scan&parse '*)
(define just-scan '*)

(define parsegen
  (lambda (the-grammar)
    (sllgen:make-define-datatypes the-lexer the-grammar)
    (set! scan&parse
      (sllgen:make-string-parser the-lexer the-grammar))
    (set! just-scan
      (sllgen:make-string-scanner the-lexer the-grammar))))

(parsegen grammar-with-inheritance)

(define run
  (lambda (string)
    (eval-program (scan&parse string))))


;;;;;;;;;;;;;;;; the interpreter ;;;;;;;;;;;;;;;;

(define eval-program 
  (lambda (pgm)
    (cases program pgm
      (a-program (class-decls body)
        (elaborate-class-decls! class-decls) ; new for ch6
        (eval-expression body (init-env))))))

(define eval-expression
  (lambda (exp env)
    (cases expression exp
      (lit-exp (datum) datum)
     (var-exp (id) (apply-env env id))
      (primapp-exp (prim rands)
        (let ((args (eval-rands rands env)))
          (apply-primitive prim args)))
      (if-exp (test-exp true-exp false-exp)
        (if (true-value? (eval-expression test-exp env))
          (eval-expression true-exp env)
          (eval-expression false-exp env)))
      (let-exp (ids rands body)
	(let ((args (eval-rands rands env)))
          (eval-expression body (extend-env ids args env))))
      (proc-exp (ids body)
        (closure ids body env))
      (app-exp (rator rands)
        (let ((proc (eval-expression rator env))
              (args (eval-rands      rands env)))
          (if (procval? proc)
            (apply-procval proc args)
            (error 'eval-expression 
              "attempt to apply non-procedure ~s"
              proc))))
      (letrec-exp (proc-names idss bodies letrec-body)
        (eval-expression letrec-body
          (extend-env-recursively proc-names idss bodies env)))
      (varassign-exp (id rhs-exp)
        (set-location!
          (apply-env-location env id)
          (eval-expression rhs-exp env))
        1)
      (begin-exp (exp1 exps)
        (let loop ((acc (eval-expression exp1 env))
                   (exps exps))
          (if (null? exps) acc
            (loop (eval-expression (car exps) env) (cdr exps)))))

;;;;;;;;;;;;;;;; new cases for chap 6 ;;;;;;;;;;;;;;;;

      (new-object-exp (class-name rands)
        (let ((args (eval-rands rands env))
;              (obj (new-object (lookup-class class-name)))
              (obj (new-object class-name)))
          (find-method-and-apply 'initialize class-name obj args)
          obj))

      (method-app-exp (obj-exp method-name rands)
        (let ((args (eval-rands rands env))
              (obj (eval-expression obj-exp env)))
          (find-method-and-apply
            method-name (object->class-name obj) obj args)))

      (super-call-exp (method-name rands)
        (let ((args (eval-rands rands env))
              (obj (apply-env env 'self)))
          (find-method-and-apply method-name
	    (apply-env env '%super) obj args)))

;;;;;;;;;;;;;;;; end new cases for chap 6 ;;;;;;;;;;;;;;;;

      )))
      

(define eval-rands
  (lambda (exps env)
    (map
      (lambda (exp) (eval-expression exp env))
      exps)))

(define apply-primitive
  (lambda (prim args)
    (cases primitive prim
      (add-prim  () (+ (car args) (cadr args)))
      (subtract-prim () (- (car args) (cadr args)))
      (mult-prim  () (* (car args) (cadr args)))
      (incr-prim  () (+ (car args) 1))
      (decr-prim  () (- (car args) 1))
      (zero-test-prim () (if (zero? (car args)) 1 0))
      (list-prim () args)               ;already a list
      (nil-prim () '())
      (car-prim () (car (car args)))
      (cdr-prim () (cdr (car args)))
      (cons-prim () (cons (car args) (cadr args)))
      (null?-prim () (if (null? (car args)) 1 0))
      )))

(define init-env 
  (lambda ()
    (extend-env
      '(i v x)
      '(1 5 10)
      (empty-env))))

;;;;;;;;;;;;;;;; booleans ;;;;;;;;;;;;;;;;

(define true-value?
  (lambda (x)
    (not (zero? x))))


;;;;;;;;;;;;;;;; declarations ;;;;;;;;;;;;;;;;

; (define let-decl->id
;   (lambda (d)
;     (cases let-decl d
;       (a-let-decl (id exp) id))))

; (define let-decl->rand
;   (lambda (d)
;     (cases let-decl d
;       (a-let-decl (id exp) exp))))

; (define proc-decl->name
;   (lambda (pd)
;     (cases proc-decl pd
;       (a-proc-decl (name ids body) name))))

; (define proc-decl->ids
;   (lambda (pd)
;     (cases proc-decl pd
;       (a-proc-decl (name ids body) ids))))

; (define proc-decl->body
;   (lambda (pd)
;     (cases proc-decl pd
;       (a-proc-decl (name ids body) body))))

(define class-decl->class-name
  (lambda (the-class-decl)
    (cases class-decl the-class-decl
      (a-class-decl (class-name super-name field-ids method-decls)
	class-name))))

(define class-decl->super-name
  (lambda (the-class-decl)
    (cases class-decl the-class-decl
      (a-class-decl (class-name super-name field-ids method-decls)
	super-name))))

(define class-decl->field-ids
  (lambda (the-class-decl)
    (cases class-decl the-class-decl
      (a-class-decl (class-name super-name field-ids method-decls)
        field-ids))))

(define class-decl->method-decls
  (lambda (the-class-decl)
    (cases class-decl the-class-decl
      (a-class-decl (class-name super-name field-ids method-decls)
        method-decls))))

; (define class-decl->field-ids
;   (lambda (cd)
;     (map field-decl->name (class-decl->field-decls cd))))

; (define field-decl->name
;   (lambda (fd)
;     (cases field-decl fd
;       (a-field-decl (field-id) field-id))))

; (define field-decls->names
;   (lambda (fds)
;     (map field-decl->name fds)))

(define method-decl->method-name
  (lambda (md)
    (cases method-decl md
      (a-method-decl (method-name ids body) method-name))))

(define method-decl->ids
  (lambda (md)
    (cases method-decl md
      (a-method-decl (method-name ids body) ids))))

(define method-decl->body
  (lambda (md)
    (cases method-decl md
      (a-method-decl (method-name ids body) body))))

(define method-decls->method-names
  (lambda (mds)
    (map method-decl->method-name mds)))
	
;;;;;;;;;;;;;;;; procedures ;;;;;;;;;;;;;;;;

(define-datatype procval procval?
  (closure 
    (ids (list-of symbol?)) 
    (body expression?)
    (env environment?)))

(define apply-procval
  (lambda (proc args)
    (cases procval proc
      (closure (ids body env)
        (eval-expression body (extend-env ids args env))))))

;;;;;;;;;;;;;;;; type constructors ;;;;;;;;;;;;;;;;

(define vector-of			; doesn't test contents
  (lambda (pred)
    (lambda (x) (vector? x))))
	       
;;;;;;;;;;;;;;;; environments ;;;;;;;;;;;;;;;;

;; Mutable environment structure from chapter 3 ala cth.

;;; {fig:denval-ADT}

(define-datatype location location?
  (a-location
    (position integer?)
    (vec vector?)))

(define location->contents
  (lambda (loc)
    (cases location loc
      (a-location (position vec)
        (vector-ref vec position)))))

(define set-location!
  (lambda (loc val)
    (cases location loc
      (a-location (position vec)
        (vector-set! vec position val)))))

;;; {fig:mutable-environments}

(define-datatype environment environment?
  (empty-env-record)
  (extended-env-record
    (syms (list-of symbol?))
    (vec vector?)              ; can use this for anything.
    (env environment?)))


(define empty-env
  (lambda ()
    (empty-env-record)))

(define extend-env
  (lambda (syms vals env)
    (extended-env-record syms (list->vector vals) env)))

(define extend-env-recursively
  (lambda (proc-names idss bodies old-env)
    (let ((len (length proc-names)))
      (let ((vec (make-vector len)))
        (let ((env (extended-env-record proc-names vec old-env)))
          (for-each
            (lambda (pos ids body)
              (vector-set! vec pos (closure ids body env)))
            (iota len) idss bodies)
          env)))))

(define apply-env-location
  (lambda (env sym)
    (cases environment env
      (empty-env-record ()
        (error 'apply-env-location "no association for symbol ~s"
          sym))
      (extended-env-record (syms vec env)
        (let ((position (env-find-position sym syms)))
          (if (number? position)
            (report-location position vec)
            (apply-env-location env sym)))))))


(define apply-env
  (lambda (env sym)
    (location->contents
      (apply-env-location env sym))))

(define env-find-position 
  (lambda (sym los)
    (list-find-position sym los)))

;; this could be inlined.
(define report-location 
  (lambda (pos vec)
    (a-location pos vec)))

(define list-find-position
  (lambda (sym los)
    (list-index (lambda (sym1) (eqv? sym1 sym)) los)))

;; waiting for 6-3.  Brute force code.
(define list-find-last-position
  (lambda (sym los)
    (let loop
      ((los los) (curpos 0) (lastpos #f))
      (cond
        ((null? los) lastpos)
        ((eqv? sym (car los))
         (loop (cdr los) (+ curpos 1) curpos))
        (else (loop (cdr los) (+ curpos 1) lastpos))))))

(define list-index
  (lambda (pred ls)
    (cond
      ((null? ls) #f)
      ((pred (car ls)) 0)
      (else (let ((list-index-r (list-index pred (cdr ls))))
              (if (number? list-index-r)
                (+ list-index-r 1)
                #f))))))

(define iota
  (lambda (end)
    (let loop ((next 0))
      (if (>= next end) '()
        (cons next (loop (+ 1 next)))))))

(define difference
  (lambda (set1 set2)
    (cond
      ((null? set1) '())
      ((memq (car set1) set2)
       (difference (cdr set1) set2))
      (else (cons (car set1) (difference (cdr set1) set2))))))

;;; additions for interp6-1:

(define extend-env-with-vector
  (lambda (syms vec env)
    (extended-env-record syms vec env)))

;;;;;;;;;;;;;;;; test harness ;;;;;;;;;;;;;;;;

(define test-list
  (lambda (groups tests)
    (let ((bugs '()))
      (for-each 
        (lambda (test-item)
          (let ((description   (car test-item))
                (pgm          (cadr test-item))
                (correct-ans (caddr test-item))
                (group      (cadddr test-item)))
            (if (memq group groups)
              (begin
                (printf "test: ~a~%~a~%" description pgm)
                (let ((ans (run pgm)))
                  (printf "~a~%" ans)
                  (if (equal? ans correct-ans)
                    (printf "correct~%~%")
                    (begin
                      (printf "incorrect~%~%")
                      (set! bugs
                        (cons description bugs)))))))))
        tests)
      (if (null? bugs)
        (printf "no bugs found~%")
        (printf "incorrect answers on tests: ~a~%" bugs)))))

(define all-tests '())

;; override test of the same name, otherwise append.
(define add-test
  (lambda (group name program correct-answer)
    (set! all-tests
      (let loop ((tests all-tests))
	(cond 
	  ((null? tests) (list (list name program correct-answer group)))
	  ((eq? (caar tests) name)
	   (cons (list name program correct-answer group) (cdr tests)))
	  (else (cons (car tests) (loop (cdr tests)))))))))

(define-syntax define-test
  (syntax-rules ()
    ((_ ?name ?program ?ans)
     (add-test 'name program 'ans))))

(define test-groups
  (lambda (groups) 
    (test-list groups all-tests)))

(define run-test
  (lambda (name)
    (let ((test-item (assq name all-tests)))
      (let ((group (cadddr test-item)))
        (test-list (list group) (list (assq name all-tests)))))))

;;;;;;;;;;;;;;;; tests ;;;;;;;;;;;;;;;;

(add-test 'functional 'pgm1 "1" 1)

(add-test 'functional 'pgm2 "add1(x)" 11)

(add-test 'functional 'primitive-app-2 "+(x,3)" 13)

(add-test 'functional 'pgm3 "if zero?(x) then 3 else 4" 4)

(add-test 'functional 'pgm4 "if zero?(0) then 3 else 4" 3)

(add-test 'functional 'pgm5 "let x = 3 y = 4 in +(x,y)" 7)

(add-test 'functional 'pgm6 "
let fix =  proc (f)
            let d = proc (x) proc (z) (f (x x) z)
            in proc (n) (f (d d) n)
    t4m = proc (f,x) if x then +(4,(f -(x,1))) else 0
in let times4 = (fix t4m)
   in (times4 3)" 12)

(add-test 'functional 'bind-a-proc "let f = proc(x,y)+(x,y) in (f 3 4)" 7)
(add-test 'functional 'apply-a-proc "(proc(x,y)-(x,y)  4 3)" 1)


(add-test 'functional 'fact-of-6  "letrec
  fact(x) = if zero?(x) then 1 else *(x, (fact sub1(x)))
in (fact 6)" 720)

(add-test 'functional 'odd-of-13  "letrec
         even(x) = if zero?(x) then 1 else (odd sub1(x))
         odd(x)  = if zero?(x) then 0 else (even sub1(x))
       in (odd 13)" 1)

(add-test 'functional 'HO-nested-letrecs
  "letrec even(odd,x) =  if zero?(x) then 1 else (odd sub1(x))
   in letrec  odd(x)  = if zero?(x) then 0 else (even odd sub1(x))
   in (odd 13)" 1)

(add-test 'functional 'gensym-test
"let g = let count = 0 in proc() 
                        let d = set count = add1(count)
                        in count
in +((g), (g))"
3)

(add-test 'functional 'even-odd-via-set "
let x = 0
in letrec even() = if zero?(x) then 1 
                                  else let d = set x = sub1(x)
                                       in (odd)
              odd()  = if zero?(x) then 0 
                                  else let d = set x = sub1(x)
                                       in (even)
   in let d = set x = 13 in (odd)" 1)

(add-test 'oop-with-inh 'create-empty-class
  "class c1 extends object  3" 3)

(add-test 'oop-with-inh 'create-class-with-method "
class c1 extends object 
  field y 
  method gety()y 33 "
33)

(add-test 'oop-with-inh 'create-object "
class c1 extends object  
 method initialize()0 
let o1 = new c1() in 11
" 11)


(add-test 'oop-with-inh 'send-msg-1 "
class c1 extends object 
  field s 
  method initialize()set s = 44
  method gets()s
  method sets(v)set s = v
  
let o1 = new c1() in send o1 gets()
"
44)

(add-test 'oop-with-inh 'send-msg-2 "
class c1 extends object 
  field s 
  method initialize()set s = 44
  method gets()s
  method sets(v)set s = v
  
let o1 = new c1() 
    t1 = 0
    t2 = 0 
in begin
     set t1 = send o1 gets();
     send o1 sets(33);
     set t2 = send o1 gets();
     list(t1, t2)
  end
"
'(44 33))

(add-test 'oop-with-inh 'test-self-1 "
class c extends object 
  field s
  method initialize(v)set s = v
  method sets(v)set s = v
  method gets()s
  method testit()send self sets(13)
  
let o = new c (11)
       t1 = 0
       t2 = 0
   in begin 
       set t1 = send o gets();
       send o testit();
       set t2 = send o gets();
       list(t1,t2)
      end" '(11 13))

(add-test 'oop-with-inh 'two-queues "
class queue extends object 
  field q_in
  field q_out
  field ans
  method initialize()send self reset()
  method reset() 
   begin 
    set q_in = nil();
    set q_out = nil();
    send self countup()
   end
   
   method empty?()if null?(q_in) then null?(q_out)
                                  else 0
   method enq(x)begin
                  send self countup();
                  set q_in = cons(x,q_in)
                 end
   method deq()
     letrec reverse(l) = (reverse_help l nil())
                 reverse_help(inp,out) = if null?(inp) then out
                                         else (reverse_help 
                                                 cdr(inp) cons(car(inp), out))
      in if send self empty?() then 0
                                else begin
                                      send self countup();
                                      if null?(q_out) then
                                        begin set q_out = (reverse q_in);
                                              set q_in = nil()
                                        end
                                        else 0;
                                      set ans = car(q_out);
                                      set q_out = cdr(q_out);
                                      ans
                                     end 
      method countup()1    % stub
      method get_total()1  % stub
                   
let o1 = new queue ()
    o2 = new queue ()
    t1 = 0 t2 = 0 t3 = 0
    t4 = 0 t5 = 0 t6 = 0
    tot1 = 0 tot2 = 0
in begin
       send o1 enq(11);
       send o2 enq(21);
       send o1 enq(12);
       send o2 enq(22);
       set t1 = send o1 deq();
       set t2 = send o1 deq();
       set t3 = send o2 deq();
       set t4 = send o2 deq();
       set t5 = send o1 get_total();
       set t6 = send o2 get_total();
       list(t1,t2,t3,t4,t5,t6)
  end" '(11 12 21 22 1 1))

;; next one is queue with shared counter object (passed at initialization)

(add-test 'oop-with-inh 'counter-1 "
class counter extends object 
 field count
 method initialize()set count = 0
 method countup()set count = add1(count)
 method getcount()count
 
let o1 = new counter ()
    t1 = 0
    t2 = 0
in begin
    set t1 = send o1 getcount();
    send o1 countup();
    set t2 = send o1 getcount();
    list(t1,t2)
   end
" '(0 1))

(add-test 'oop-with-inh 'shared-counter-1 "
class counter extends object
  field count
   method initialize()set count = 0
   method countup()set count = add1(count)
   method getcount()count
   
class c1 extends object 
   field n
   field counter1
   method initialize(a_counter)
    begin
     set n = 0;
     set counter1 = a_counter
    end
   method countup()
     begin
      send counter1 countup();
      set n = add1(n)
     end
   method getstate()list(n, send counter1 getcount())
   
let counter1 = new counter()
in let o1 = new c1(counter1)
       o2 = new c1(counter1)
in begin
     send o1 countup();
     send o2 countup();
     send o2 countup();
     list( send o1 getstate(),
           send o2 getstate())
   end
" '((1 3) (2 3)))

(add-test 'oop-with-inh 'two-queues-with-counter "
class counter extends object
  field c_count
   method initialize()set c_count = 0
   method countup()set c_count = add1(c_count)
   method getcount()c_count
   
class queue extends object
  field q_in 
   field q_out
   field ans
   field count
   method initialize(the_counter)
    begin
      set count = the_counter;  % must do this first, because reset counts.
      send self reset()
     end
    
   method reset()begin set q_in = nil();
                        set q_out = nil();
                        send self countup()
                  end
   method empty?()if null?(q_in) then null?(q_out)
                                  else 0
   method enq(x)begin
                  send self countup();
                  set q_in = cons(x,q_in)
                 end
   method deq()
     letrec reverse(l) = (reverse_help l nil())
                 reverse_help(inp,out) = if null?(inp) then out
                                         else (reverse_help 
                                                 cdr(inp) cons(car(inp), out))
      in if send self empty?() then 0
                                else begin
                                      send self countup();
                                      if null?(q_out) then
                                        begin set q_out = (reverse q_in);
                                              set q_in = nil()
                                        end
                                        else 0;
                                      set ans = car(q_out);
                                      set q_out = cdr(q_out);
                                      ans
                                     end
      method countup()send count countup()
      method get_total()send count getcount()
   
let counter1 = new counter() in
let o1 = new queue (counter1)
    o2 = new queue (counter1)
    t1 = 0 t2 = 0 t3 = 0
    t4 = 0 t5 = 0 t6 = 0
    tot1 = 0 tot2 = 0
in begin
       send o1 enq(11);
       send o2 enq(21);
       send o1 enq(12);
       send o2 enq(22);
       set t1 = send o1 deq();
       set t2 = send o1 deq();
       set t3 = send o2 deq();
       set t4 = send o2 deq();
       set t5 = send o1 get_total();
       set t6 = send o2 get_total();
       list(t1,t2,t3,t4,t5,t6)
  end" '(11 12 21 22 10 10))


; Chris's first example
; program
;   let a = new aclass init(3)
;   in call a m(2)
;   class aclass (i)
;     method init(x) fieldassign aclass this i = x;
;     method m(y) +(field aclass this i, y)
;   end
; end

(add-test 'oop-with-inh 'chris-1 "
class aclass extends object 
  field i
  method initialize(x) set i = x
  method m(y) +(i,y)
  
let o1 = new aclass(3)
in send o1 m(2)"
5)

(add-test 'oop-with-inh 'for-book-1 "
class c1 extends object
  field i
  field j
  method initialize(x) begin set i = x; set j = -(0,x) end
  method countup(d) begin set i = +(i,d); set j = -(j,d) end
  method getstate()list(i,j)
  
let o1 = new c1(3)
    t1 = 0
    t2 = 0
in begin
    set t1 = send o1 getstate();
    send o1 countup(2);
    set t2 = send o1 getstate();
    list(t1,t2)
   end"
'((3 -3) (5 -5)))


(add-test 'oop-with-inh 'odd-even-via-self "
class oddeven extends object 
  method initialize()1
  method even(n)if zero?(n) then 1 else send self odd(sub1(n))
  method odd(n) if zero?(n) then 0 else send self even(sub1(n))
  
let o1 = new oddeven() in send o1 odd(13)"
1)

;;; inheritance starts here

(add-test 'oop-with-inh 'inherit-1 "
class c1 extends object 
  field ivar1
  method initialize()set ivar1 = 1
  
class c2 extends c1 
  field ivar2
  method initialize() 
   begin
    super initialize(); 
    set ivar2 = 1
   end
  method setiv1(n)set ivar1 = n
  method getiv1()ivar1
  
let o = new c2 ()
    t1 = 0
in begin
       send o setiv1(33);
       send o getiv1()
   end                      
" 33)

(add-test 'oop-with-inh 'inherit-2 "
class c1 extends object 
  field ivar1
  method initialize()set ivar1 = 1

  method setiv1(n)set ivar1 = n
  method getiv1()ivar1

  method foo()1
  method call-foo-from-superclass()send self foo()

  
class c2 extends c1 
  field ivar2
  method initialize() 
   begin super initialize(); set ivar2 = 1 end
   

  method foo()2

  method setiv2(n)set ivar2 = n
  method getiv2()ivar2

  method self-and-super-foo()
    list( send self foo(),  super foo())

  method test-self-from-super()
     super call-foo-from-superclass()

   
let o = new c2 ()
    t1 = 0 t2 = 0 t3 = 0 t4 = 0
in begin
       send o setiv1(33);
       list(
         send o getiv1(),
         send o self-and-super-foo(),
         send o call-foo-from-superclass(),
         send o test-self-from-super()
         )
      end                      
" '(33 (2 1) 2 2))

;     (inherit-2 ,inherit-2 (33 (2 1) 2  2 22 33))

(add-test 'oop-with-inh 'inherit-3 "
class c1 extends object 
  method initialize()1
  method m1()1
  
class c2 extends c1 
  method m1()super m1()
  method m2()2
  
class c3 extends c2 
  method m1()3
  method m2()super m2()
  method m3()super m1()
  
let o = new c3 ()
in list( send o m1(),
         send o m2(),
         send o m3()
        )
" '(3 2 1))

; Chris's first example
; program
;   let a = new aclass init(3)
;   in call a m(2)
;   class aclass (i)
;     method init(x) fieldassign aclass this i = x;
;     method m(y) +(field aclass this i, y)
;   end
; end


(add-test 'oop-with-inh 'chris-2 "
class c1 extends object 
  method initialize() 1
  method ma()1
  method mb()send self ma()
  
class c2 extends c1   % just use c1's initialize
  method ma() 2
  
let x = new c2 ()
in list(send x ma(),send x mb())
"
'(2 2))


(add-test 'oop-with-inh 'for-book-2 "
class c1 extends object 
  method initialize()1
  method m1()1
  method m2()100
  method m3()send self m2()
  
class c2 extends c1 
  method m2()2
  
let o1 = new c1()
    o2 = new c2()
in list(send o1 m1(),		% returns 1
        send o1 m2(),           % returns 100
        send o1 m3(),           % returns 100
        send o2 m1(),           % returns 1 (from c1)
        send o2 m2(),           % returns 2 (from c2)
        send o2 m3()            % returns 2 (c1's m3 calls c2's m2)
       )
"
'(1 100 100 1 2 2))

(add-test 'oop-with-inh 'sum-leaves "
class tree extends object 
  method initialize()1
  
class interior_node extends tree 
  field left
  field right
  method initialize(l,r)
   begin
    set left = l; set right = r
   end
  method sum()+(send left sum(), send right sum())
  
class leaf_node extends tree 
  field value
  method initialize(v)set value = v
  method sum()value
  
let o1 = new interior_node (
          new interior_node (
            new leaf_node(3),
            new leaf_node(4)),
          new leaf_node(5))
in send o1 sum()
"
12)

(add-test 'oop-with-inh 'check-shadowed-fields "
class c1 extends object 
  field x
  field y
  method initialize(v) begin set x = v; set y = 0 end
  method m1() x

class c2 extends c1 
  field x
  method initialize(v1,v2) begin set x = v2; 
                                    super initialize(v1) end
  method m2()list(x,y)

class c3 extends c2 
  field x
  method initialize(v1,v2,v3) begin set x = v3; 
                                       super initialize(v1,v2)
                                 end
  method m3()x

let o = new c3(1,2,3)
in list (send o m1(), send o m2(), send o m3())
"
'(1 (2 0) 3))

(add-test 'oop-with-inh 'static-super "
class c1 extends object
 method initialize () 1
 method m2() send self m3()
 method m3() 13
class c2 extends c1
 method m2() 22
 method m3() 23
 method m1() super m2()
class c3 extends c2
 method m2() 32
 method m3() 33
let o3 = new c3()
in send o3 m1()"
33)


(add-test 'oop-with-inh 'every-concept "
class a extends object
  field i
  field j
  method initialize() 1
  method setup()
    begin
      set i = 15;
      set j = 20;
      50
    end    
  method f() send self g()
  method g() +(i,j)

class b extends a
  field j
  field k
  method setup()
    begin
      set j = 100;
      set k = 200;
      super setup();
      send self h()
    end
  method g()
    list(i,j,k)
  method h() super g()

class c extends b
  method g() super h()
  method h() +(k,j)

let p = proc(o)
         let u = send o setup ()
         in list(u,
                 send o g(),
                 send o f())
in list((p new a()),
        (p new b()),
        (p new c()))
"
'((50 35 35) (35 (15 100 200) (15 100 200)) (300 35 35))
)


