;; mutable-pairs1.scm

(let ((time-stamp "Time-stamp: <2004-03-06 16:19:40 wand>"))
  (eopl:printf
    "mutable-pairs1.scm  - interp with mutable pairs ~a~%"
    (substring time-stamp 13 29)))

;;;;;;;;;;;;;;;; top level and tests ;;;;;;;;;;;;;;;;

(define run
  (lambda (string)
    (eval-program (scan&parse string))))

(define run-all
  (lambda ()
    (run-experiment run use-execution-outcome
      '(lang3-5 lang3-6 lang3-7 mutable-pairs) all-tests equal?)))

(define run-one
  (lambda (test-name)
    (run-test run test-name)))

;;;;;;;;;;;;;;;; grammatical specification ;;;;;;;;;;;;;;;;

(define the-lexical-spec
  '((whitespace (whitespace) skip)
    (comment ("%" (arbno (not #\newline))) skip)
    (identifier
      (letter (arbno (or letter digit "_" "-" "?")))
      symbol)
    (number (digit (arbno digit)) number)))

(define the-grammar
  '((program (expression) a-program)
    (expression (number) lit-exp)
    (expression (identifier) var-exp)   
    (expression
      (primitive "(" (separated-list expression ",") ")")
      primapp-exp)
    (expression
      ("if" expression "then" expression "else" expression)
      if-exp)
   (expression
      ("let" (arbno  identifier "=" expression) "in" expression)
      let-exp)
    (expression
      ("proc" "(" (separated-list identifier ",") ")" expression)
      proc-exp)
    (expression
      ("(" expression (arbno expression) ")")
      app-exp)
    (expression ("set" identifier "=" expression) varassign-exp) ; new for 3-7
    (expression                         
      ("letrec"
        (arbno identifier "(" (separated-list identifier ",") ")"
          "=" expression)
        "in" expression)
      letrec-exp)

    (primitive ("+")     add-prim)
    (primitive ("-")     subtract-prim)
    (primitive ("*")     mult-prim)
    (primitive ("add1")  incr-prim)
    (primitive ("sub1")  decr-prim)
    (primitive ("zero?") zero-test-prim)
    
;; new for mutable pairs:

    (primitive ("pair")     pair-prim)
    (primitive ("left")     left-prim)
    (primitive ("right")    right-prim)
    (primitive ("setleft")  setleft-prim)
    (primitive ("setright") setright-prim)

    ))

(sllgen:make-define-datatypes the-lexical-spec the-grammar)

(define list-the-datatypes
  (lambda () (sllgen:list-define-datatypes the-lexical-spec the-grammar)))

(define scan&parse
  (sllgen:make-string-parser the-lexical-spec the-grammar))

(define just-scan
  (sllgen:make-string-scanner the-lexical-spec the-grammar))

;;;;;;;;;;;;;;;; the interpreter ;;;;;;;;;;;;;;;;

(define eval-program 
  (lambda (pgm)
    (cases program pgm
      (a-program (body)
        (eval-expression body (init-env))))))

(define eval-expression ;^exp x env -> expval
  (lambda (exp env)
    (cases expression exp
      (lit-exp (datum) datum)
      (var-exp (id) (deref (apply-env env id)))
      (varassign-exp (id rhs-exp)  ;\new6
        (begin
          (setref! 
            (apply-env env id)
            (eval-expression rhs-exp env))
          1))
      (primapp-exp (prim rands)
        (let ((args (eval-rands rands env)))
          (apply-primitive prim args)))
      (if-exp (test-exp true-exp false-exp) ;\new4
        (if (true-value? (eval-expression test-exp env))
          (eval-expression true-exp env)
          (eval-expression false-exp env)))
      (proc-exp (ids body) (closure ids body env)) ;\new1
      (let-exp (ids rands body) ;\new3
        (let ((args (eval-rands rands env)))
          (eval-expression body (extend-env ids args env))))
      (app-exp (rator rands) ;\new7
        (let ((proc (eval-expression rator env))
              (args (eval-rands rands env)))
          (if (procval? proc)
            (apply-procval proc args)
            (eopl:error 'eval-expression
              "Attempt to apply non-procedure ~s" proc))))
      (letrec-exp (proc-names idss bodies letrec-body) ;\new4
        (eval-expression letrec-body
          (extend-env-recursively
            proc-names idss bodies env)))    
      )))

(define eval-rands
  (lambda (rands env)
    (map (lambda (x) (eval-rand x env)) rands)))

(define eval-rand
  (lambda (rand env)
    (eval-expression rand env)))

(define apply-primitive
  (lambda (prim args)
    (cases primitive prim
      (add-prim () (+ (car args) (cadr args)))
      (subtract-prim () (- (car args) (cadr args)))
      (mult-prim () (* (car args) (cadr args)))
      (incr-prim () (+ (car args) 1))
      (decr-prim () (- (car args) 1))
      (zero-test-prim () (if (zero? (car args)) 1 0))

      (pair-prim () (make-pair (car args) (cadr args)))
      (left-prim () 
        (if (pairval? (car args))
          (left (car args))
          (error 'apply-primitive "non-pair to left: ~s" (car args))))
      (right-prim ()
        (if (pairval? (car args))
          (right (car args))
          (error 'apply-primitive "non-pair to right: ~s" (car args))))
      (setleft-prim ()
        (if (pairval? (car args))
          (setleft (car args) (cadr args))
          (error 'apply-primitive "non-pair to setleft: ~s" (car args))))
      (setright-prim ()
        (if (pairval? (car args))
          (setright (car args) (cadr args))
          (error 'apply-primitive "non-pair to setright: ~s" (car args))))
      )))

(define init-env 
  (lambda ()
    (extend-env
      '(i v x)
      '(1 5 10)
      (empty-env))))

;;;;;;;;;;;;;;;; booleans ;;;;;;;;;;;;;;;;

(define true-value?
  (lambda (x)
    (not (zero? x))))

;;;;;;;;;;;;;;;; procedures ;;;;;;;;;;;;;;;;

(define-datatype procval procval?
  (closure 
    (ids (list-of symbol?)) 
    (body expression?)
    (env environment?)))

(define apply-procval
  (lambda (proc args)
    (cases procval proc
      (closure (ids body env)
        (eval-expression body (extend-env ids args env))))))
               
;;;;;;;;;;;;;;;; references ;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;; references and the store ;;;;;;;;;;;;;;;;

;;; world's dumbest model of the store:  the store is a list and a
;;; reference is number which denotes a position in the list.

(define empty-store
  (lambda () '()))

(define initialize-store!
  (lambda ()
    (set! the-store (empty-store))))

(define reference? integer?)

(define newref
  (lambda (val)
    (let ((next-ref (length the-store)))
      (set! the-store
        (append the-store (list val)))  ; YUCK!
      next-ref))) 

(define deref 
  (lambda (loc) (list-ref the-store loc)))

(define setref!                          ; YUCK!
  (lambda (loc0 val)
    (set! the-store
      (let recur ((store the-store) (loc loc0))
        (cond
          ((null? store)
           (eopl:error 'setref
             "illegal reference ~s in store ~s"
             loc0 the-store))
          ((zero? loc)
           (cons val (cdr store)))
          (else
            (cons
              (car store)
              (recur (cdr store) (- loc 1)))))))))

;;;;;;;;;;;;;;;; pairs ;;;;;;;;;;;;;;;;

;; represent a pair as two references

(define-datatype pairval pairval?  ; "pair" conflicts w/ existing Scheme
                                  ; procedure!
  (a-pair
    (left reference?)
    (right reference?)))

(define make-pair
  (lambda (val1 val2)
    (a-pair
      (newref val1)
      (newref val2))))

(define left                       
  (lambda (p)
    (cases pairval p
      (a-pair (left right) (deref left)))))

(define right
  (lambda (p)
    (cases pairval p
      (a-pair (left right) (deref right)))))

(define setleft
  (lambda (p val)
    (cases pairval p
      (a-pair (left right) (setref! left val)))))

(define setright
  (lambda (p val)
    (cases pairval p
      (a-pair (left right) (setref! right val)))))


;;;;;;;;;;;;;;;; environments ;;;;;;;;;;;;;;;;

(define-datatype environment environment?
  (empty-env-record)
  (extended-env-record
    (syms (list-of symbol?))
    (vec vector?)              ; can use this for anything.
    (env environment?))
  )

(define empty-env
  (lambda ()
    (empty-env-record)))

(define extend-env
  (lambda (syms vals env)
    (extended-env-record syms
      (list->vector (map newref vals))                 ; !!
      env)))


(define apply-env
  (lambda (env sym)
    (cases environment env
      (empty-env-record ()
        (eopl:error 'apply-env-ref "No binding for ~s" sym))
      (extended-env-record (syms vals env)
        (let ((pos (rib-find-position sym syms)))
          (if (number? pos)
            (vector-ref vals pos)
            (apply-env env sym)))))))

(define extend-env-recursively
  (lambda (proc-names idss bodies old-env)
    (let ((len (length proc-names)))
      (let ((vec (make-vector len)))
        (let ((env (extended-env-record proc-names vec old-env)))
          (for-each
            (lambda (pos ids body)
              (vector-set! vec pos
                (newref (closure ids body env)))) ; change
            (iota len) idss bodies)
          env)))))


(define rib-find-position 
  (lambda (sym los)
    (list-find-position sym los)))

(define list-find-position
  (lambda (sym los)
    (list-index (lambda (sym1) (eqv? sym1 sym)) los)))

(define list-index
  (lambda (pred ls)
    (cond
      ((null? ls) #f)
      ((pred (car ls)) 0)
      (else (let ((list-index-r (list-index pred (cdr ls))))
              (if (number? list-index-r)
                (+ list-index-r 1)
                #f))))))

(define iota
  (lambda (end)
    (let loop ((next 0))
      (if (>= next end) '()
        (cons next (loop (+ 1 next)))))))

(define difference
  (lambda (set1 set2)
    (cond
      ((null? set1) '())
      ((memv (car set1) set2)
       (difference (cdr set1) set2))
      (else (cons (car set1) (difference (cdr set1) set2))))))

;;;;;;;;;;;;;;;; new test items ;;;;;;;;;;;;;;;;

(add-test! 'mutable-pairs 'gensym-using-mutable-pair-left
"let g = let count = pair(0,0) in proc() 
                        let d = setleft(count,add1(left(count)))
                        in left(count)
in +((g), (g))"
3)

;;; gotta check the right, too!

(add-test! 'mutable-pairs 'gensym-using-mutable-pair-right
"let g = let count = pair(0,0) in proc() 
                        let d = setright(count,add1(right(count)))
                        in right(count)
in +((g), (g))"
3)

