;;; Revision History:

(let ((time-stamp "Time-stamp: <2002-02-19 18:17:28 wand>"))
  (eopl:printf "cfa.scm: Control-Flow Analysis for mp3.scm ~a~%"
    (substring time-stamp 13 29)))

;;;;;;;;;;;;;;;; top level ;;;;;;;;;;;;;;;;

(define label-string
  (lambda (string)
    (label-program (scan&parse string))))

(define label-all
  (lambda ()
    (run-experiment
      label-string
      use-execution-outcome
      '(lang3-5)
      all-tests
      (lambda (a b) #t))))

(define label-one
  (lambda (test-name)
    (run-test label-string test-name)))


;;;;;;;;;;;;;;;; syntactic specification ;;;;;;;;;;;;;;;;

(define the-lexical-spec
  '((whitespace (whitespace) skip)
    (comment ("%" (arbno (not #\newline))) skip)
    (identifier
      (letter (arbno (or letter digit "_" "-" "?")))
      make-symbol)
    (number (digit (arbno digit)) make-number)))

(define the-grammar
  '((program (expression) a-program)
    (expression (number) lit-exp)
    (expression ("true") true-exp)
    (expression ("false") false-exp)
    (expression (identifier) var-exp)   
    (expression
      (primitive "(" (separated-list expression ",") ")")
      primapp-exp)
    (expression
      ("if" expression "then" expression "else" expression)
      if-exp)
    (expression
      ("proc" "(" (separated-list identifier ",") ")" expression)
      proc-exp)
    (expression
      ("(" expression (arbno expression) ")")
      app-exp)
    (expression
      ("let" (arbno  identifier "=" expression) "in" expression)
      let-exp)
    (expression                         ;; not implemented here
      ("letrec"
        (arbno identifier "(" (separated-list identifier ",") ")"
          "=" expression)
        "in" expression)
      letrec-exp)

    (primitive ("+")     add-prim)
    (primitive ("-")     subtract-prim)
    (primitive ("*")     mult-prim)
    (primitive ("add1")  incr-prim)
    (primitive ("sub1")  decr-prim)
    (primitive ("zero?") zero-test-prim)

    ))

(sllgen:make-define-datatypes the-lexical-spec the-grammar)

(define show-the-datatypes
  (lambda () (sllgen:list-define-datatypes the-lexical-spec the-grammar)))

(define scan&parse
  (sllgen:make-string-parser the-lexical-spec the-grammar))

(define just-scan
  (sllgen:make-string-scanner the-lexical-spec the-grammar))

;;;;;;;;;;;;;;;; the analyzer ;;;;;;;;;;;;;;;;

;;; datatype of flat-expressions

(define label? number?)

(define-datatype flat-expression-node flat-expression?
   (flat-lit-exp
     (datum number?))
   (flat-true-exp)
   (flat-false-exp)
   (flat-var-exp (id symbol?))
   (flat-primapp-exp
     (prim primitive?)
     (arg-labels (list-of label?)))
   (flat-if-exp
     (test-label label?)
     (then-label label?)
     (else-label label?))
   (flat-proc-exp
     (id-labels (list-of label?))
     (body-label label?))
   (flat-app-exp
     (rator-label label?)
     (rand-labels (list-of label?)))
   (flat-let-exp
     (id-labels (list-of label?))
     (rhs-labels (list-of label?))
     (body-label label?))
   ;; flat-letrec-exps are unimplemented here
   )


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; the labeler
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; the labeler uses a "blackboard architecture":  as it works, it
;; makes notations in a global state (the blackboard).

;; the blackboard for the labeller consists of a free label counter 
;; and a list of lists of the form (label, flat-expression-node,
;; original-expression).

(define *the-label-state* 'uninitialized)

(define initialize-the-label-state
  (lambda ()
    (set! *the-label-state* (cons 0 '()))))

(define bindings-of-the-label-state
  (lambda ()
    (cdr *the-label-state*)))

(define next-free-label-of-the-label-state
  (lambda ()
    (car *the-label-state*)))

;; add-exp adds an item to a "fresh label" in the store, and returns that
;; label.

(define add-exp                         
  (lambda (exp node)
    (let ((next-free-label (car *the-label-state*))
          (bindings        (cdr *the-label-state*)))
      (set! *the-label-state*
        (cons
          (+ 1 next-free-label)
          (cons
            (list next-free-label node exp)
            bindings)))
      next-free-label)))

;;; label-program takes a program and returns an label and a
;;; vector mapping labels to lists of the form (label, flat-expression-node,
;;; original-expression).

(define label-program 
  (lambda (pgm)
    (cases program pgm
      (a-program (body)
        (initialize-the-label-state)
        (let* ((top-label (label-expression body (empty-env)))
               (items (bindings-of-the-label-state))
               (size (length items))
               (vec (make-vector size)))
          (for-each
            (lambda (item)
              (let ((label (car item))
                    (flat-exp (cadr item))
                    (orig-exp (caddr item)))
                (vector-set! vec label item)))
            items)
          (cons top-label vec))))))

;; label-expression will label an expression and its subexpressions,
;; adding them to the label state, and return the label that it puts
;; on the whole expression.

;; here env maps variables to their binding labels

(define label-expression                ; exp x env -> comp(label)
  (lambda (exp env)
    (cases expression exp
      (lit-exp (datum) (add-exp exp (flat-lit-exp datum)))
      (var-exp (id) (apply-env env id)) ; the flat-var will be added
                                        ; by the declaration
      (true-exp () (add-exp exp (flat-true-exp)))
      (false-exp () (add-exp exp (flat-false-exp)))
      (primapp-exp (prim rands)
        (let ((arg-labels (label-expressions rands env)))
          (add-exp exp (flat-primapp-exp prim arg-labels))))
      (if-exp (test-exp true-exp false-exp)
        (let ((test-label (label-expression test-exp env))
              (true-label (label-expression true-exp env))
              (false-label (label-expression false-exp env)))
          (add-exp exp
            (flat-if-exp test-label true-label false-label))))
      (proc-exp (ids body)
        (let* ((id-labels (label-ids ids))
               (body-label
                 (label-expression body (extend-env ids id-labels env))))
          (add-exp exp (flat-proc-exp id-labels body-label))))
      (app-exp (rator rands)
        (let ((rator-label (label-expression rator env))
              (rand-labels (label-expressions rands env)))
          (add-exp exp (flat-app-exp rator-label rand-labels))))
      (let-exp (ids rands body)
        (let ((id-labels (label-ids ids))
              (arg-labels (label-expressions rands env)))
          (let ((body-label
                  (label-expression body (extend-env ids id-labels env))))
            (add-exp exp (flat-let-exp id-labels arg-labels body-label)))))
      ;; save letrec for later
      (else (eopl:error 'label-expression "Unimplemented expression: ~s" exp))
      )))

(define label-expressions
  (lambda (exps env)
    (map
      (lambda (exp)
        (label-expression exp env))
      exps)))

(define label-ids
  (lambda (ids)
    (map
      (lambda (id) (add-exp (var-exp id) (flat-var-exp id)))
      ids)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; generate constraints
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-datatype absval absval?
  (closure-absval (origin label?))
;   (scalar-absval (origin label?))
;   (bool-absval   (origin label?))
;   (error-absval  (origin label?))
  )

;;; representation of constraints

(define-datatype constraint constraint?
  (membership-constraint     ;; represents absval in Phi(setname)
    (absval absval?)
    (setname label?))
  (subset-constraint       ;; represents Phi(setname1) \subset Phi(setname2)
    (setname1 label?)
    (setname2 label?))
  (conditional-constraint  ;; represents absval in Phi(setname) => conseq
    (absval absval?)
    (setname label?)
    (consequent constraint?)))

;; the generator will also use a blackboard architecture.  
;; The state of the generator will be a list of constraints.
;; The value of the global variable assert-constraint! will be a
;; procedure for adding a constraint to the blackboard.

(define *the-generator-state* 'uninitialized)

(define the-generator-state
  (lambda ()
    *the-generator-state*))

(define add-constraint!
  (lambda (constraint)
    (set! *the-generator-state*
      (cons constraint *the-generator-state*))))

(define initialize-the-generator-state
  (lambda ()                           
    (set! *the-generator-state* '())
    (set! assert-constraint! add-constraint!)))

;; takes a vector of nodes and finds the constraints, using
;; assert-constraint! 

(define constraints-of
  (lambda (vec return-fn)
    (generate-linear-constraints! vec)
    (generate-quadratic-constraints! vec)
    (return-fn)))
    
;; a handy top-level composite:

(define generate-constraints
  (lambda (str)
    (let ((vec (cdr (label-program (scan&parse str)))))
      (pretty-print vec)
      (initialize-the-generator-state)
      (constraints-of vec the-generator-state))))

(define generate-linear-constraints!
  (lambda (vec)
    (vector-for-eachx! vec
      (lambda (item)
        (let ((label (car item)) (node (cadr item)))
          (cases flat-expression-node node
            (flat-proc-exp (id-labels body-labels)
              (assert-constraint!
                (membership-constraint
                  (closure-absval label)
                  label)))
            (else '())))))))

(define generate-quadratic-constraints!
  (lambda (vec)
    (vector-for-eachx! vec
      (lambda (item)
        (let ((app-label (car item)) (app-node (cadr item)))
          (cases flat-expression-node app-node
            (flat-app-exp (rator-label rand-labels)
              ;; we've found an application, now let's look for
              ;; abstractions
              (vector-for-eachx! vec
                (lambda (item)
                  (let ((lambda-label (car item))
                        (lambda-node (cadr item)))
                    (cases flat-expression-node lambda-node
                      (flat-proc-exp (id-labels body-label)
                        (for-each
                          assert-constraint!
                          (cross-constraints
                            app-label lambda-label
                            rator-label rand-labels
                            id-labels body-label)))
                      (else '()))))))
            (else '())))))))

(define cross-constraints
  (lambda (app-label lam-label
            rator-label rand-labels
            ids-labels body-label)
    (if (= (length rand-labels) (length ids-labels))
      ;; if argument lengths match, then go ahead
      (cons
        (conditional-constraint
          (closure-absval lam-label) rator-label
          (subset-constraint body-label app-label))
        (map
          (lambda (rand-label id-label)
            (conditional-constraint
              (closure-absval lam-label) rator-label
              (subset-constraint rand-label id-label)))
          rand-labels ids-labels))
      ;; right now we don't track errors
      '())))

(define vector-for-eachx!
  (lambda (vec fn)
    (let ((len (vector-length vec)))
      (let loop ((i 0))
        (if (eqv? i len)
          '()
          (begin
            (fn (vector-ref vec i))
            (loop (+ 1 i))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; solving the constraints
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; We'll invoke the solver as we go along.
;; We'll do this by using a different value of assert-constraint!.

(define solve-constraints
  (lambda (str)
    (let ((vec (cdr (label-program (scan&parse str)))))
      (pretty-print vec)
      (initialize-the-solver-state (vector-length vec))
      ;; since constraints-of calls assert-constraint!, constraints-of
      ;; now solves the constraints as it goes along.
      (constraints-of vec return-the-solution))))

;; Now, not only do we add c to the generator state, but we draw any
;; conclusions we can from it.

(define assert-constraint-and-draw-conclusions!
    (lambda (c)
      (set! *the-generator-state*
        (cons c *the-generator-state*))
      (cases constraint c
        (membership-constraint (absval set) (assert-member! absval set))
        (subset-constraint (set1 set2) (assert-subset! set1 set2))
        (conditional-constraint (absval set conseq)
          (if (member-of? absval set)
            (assert-constraint! conseq)
            (add-conditional! absval set conseq))))))

(define assert-member!
  (lambda (absval set)
    (if (not (member-of? absval set))
      (begin
        (add-member! absval set)
        (for-each-successor set
          (lambda (set2)
            (assert-member! absval set2)))
        (for-each-conditional-constraint absval set 
          (lambda (consequent)
            (assert-constraint! consequent)))))))

(define assert-subset!
  (lambda (set1 set2)
    (if (not (successor-of? set1 set2))
      (begin
        (add-successor! set1 set2)
        (for-each-member set1
          (lambda (absval) (assert-member! absval set2)))))))

;;; the state of the solver consists of 5 items:
;;; *the-generator-state* -- same as in the generator
;;; *members* -- for each i, (vector-ref *members* i) = Phi(i)
;;; *successors* -- for each i, 
;;;      (vector-ref *successors* j) 
;;;        = {j | we've deduced Phi(i) subset Phi(j)}
;;; *conditionals* -- for each i, (vector-ref *conditionals* i)
;;;        = {asserted conditionals of the form (absval in Phi(i) => ...)
;;; assert-constraint! -- set to assert-constraint-and-draw-conclusions!

(define *members* 'uninitialized)
(define *successors* 'uninitialized)
(define *conditionals* 'uninitialized)

(define initialize-the-solver-state
  (lambda (n)                           
    (set! assert-constraint! assert-constraint-and-draw-conclusions!)
    (set! *members* (make-vector n '()))
    (set! *successors* (make-vector n '()))
    (set! *conditionals* (make-vector n '()))))

(define member-of?
  (lambda (absval setname)
    ;; absval's are structured, so you need equal? (yuck)
    (member absval (vector-ref *members* setname))))

(define successor-of?
  (lambda (i j)
    (memv j (vector-ref *successors* i))))

(define add-member!                 ; assumes i not in Phi(j)
  (lambda (absval j)
    (vector-set! *members* j (cons absval (vector-ref *members* j)))))

(define add-successor!              ; assumes i not in successors(j)
  (lambda (i j)
    (vector-set! *successors* i (cons j (vector-ref *successors* i)))))

(define for-each-successor
  (lambda (set fn)
    (for-each fn (vector-ref *successors* set))))

(define for-each-member
  (lambda (set fn)
    (for-each fn (vector-ref *members* set))))

;; conditionals are only added once, so we don't have to worry about
;; whether it's already in the database.

(define add-conditional!
  (lambda (absval set conseq)
    (vector-set! *conditionals* set
      (cons
        (list absval conseq)
        (vector-ref *conditionals* set)))))

(define for-each-conditional-constraint
  (lambda (absval set fn)
    (for-each
      (lambda (l)
        (if (equal? (car l) absval) (fn (cadr l))))
      (vector-ref *conditionals* set))))

;; in fact, we could do this !without keeping the list of constraints!;
;; here's the code for unary procs:

;; (define for-each-conditional-constraint
;;   (lambda (i j fn)
;;     (if (and (abstraction? i) (operator? j))
;;      (begin
;;         (fn (operator->operand j) (bound-variable i))
;;         (fn (abs->body i)         (operator->application j))))))

(define return-the-solution
  (lambda ()
    (let ((sets (iota (vector-length *members*))))
      (append
        (map
          (lambda (set)
            (list '*members* set (vector-ref *members* set)))
          sets)
        (map
          (lambda (set)
            (list '*successors* set (vector-ref *successors* set)))
          sets)))))

;;; environments

(define-datatype environment environment?
  (empty-env-record)
  (extended-env-record
    (syms (list-of symbol?))
    (vec vector?)                   ; can use this for anything.
    (env environment?))
  )

(define empty-env
  (lambda ()
    (empty-env-record)))

(define extend-env
  (lambda (syms vals env)
    (extended-env-record syms (list->vector vals) env)))

(define apply-env
  (lambda (env sym)
    (cases environment env
      (empty-env-record ()
        (eopl:error 'apply-env "No binding for ~s" sym))
      (extended-env-record (syms vals env)
        (let ((position (env-find-position sym syms)))
          (if (number? position)
            (vector-ref vals position)
            (apply-env env sym)))))))

(define env-find-position 
  (lambda (sym los)
    (list-find-position sym los)))

(define list-find-position
  (lambda (sym los)
    (list-label (lambda (sym1) (eqv? sym1 sym)) los)))

(define list-label
  (lambda (pred ls)
    (cond
      ((null? ls) #f)
      ((pred (car ls)) 0)
      (else (let ((list-label-r (list-label pred (cdr ls))))
              (if (number? list-label-r)
                (+ list-label-r 1)
                #f))))))

(define iota
  (lambda (end)
    (let loop ((next 0))
      (if (>= next end) '()
        (cons next (loop (+ 1 next)))))))
