(let ((time-stamp "Time-stamp: <2002-04-09 18:07:25 wand>"))
  (eopl:printf "3-5lex.scm: lexical addressing for 3-5.scm ~a~%"
    (substring time-stamp 13 29)))

;;;;;;;;;;;;;;;; top level and tests ;;;;;;;;;;;;;;;;

(define run
  (lambda (string)
    (eval-program
      (translation-of-program
        (scan&parse string)))))

(define run-all
  (lambda ()
    (run-experiment run use-execution-outcome
      '(lang3-1 lang3-5) all-tests equal?)))

(define run-one
  (lambda (test-name)
    (run-test run test-name)))

;;;;;;;;;;;;;;;; grammatical specification ;;;;;;;;;;;;;;;;

(define the-lexical-spec
  '((whitespace (whitespace) skip)
    (comment ("%" (arbno (not #\newline))) skip)
    (identifier
      (letter (arbno (or letter digit "_" "-" "?")))
      symbol)
    (number (digit (arbno digit)) number)))

(define the-grammar
  '((program (expression) a-program)
    (expression (number) lit-exp)
    (expression (identifier) var-exp)   
    (expression
      (primitive "(" (separated-list expression ",") ")")
      primapp-exp)
    (expression
      ("if" expression "then" expression "else" expression)
      if-exp)
    (expression
      ("let" (arbno  identifier "=" expression) "in" expression)
      let-exp)
    (expression
      ("proc" "(" (separated-list identifier ",") ")" expression)
      proc-exp)
    (expression
      ("(" expression (arbno expression) ")")
      app-exp)

    (primitive ("+")     add-prim)
    (primitive ("-")     subtract-prim)
    (primitive ("*")     mult-prim)
    (primitive ("add1")  incr-prim)
    (primitive ("sub1")  decr-prim)
    (primitive ("zero?") zero-test-prim)
    
    (expression ("%lexvar" number number) lexvar-exp)
    (expression
      ("%let" (arbno expression) "in" expression)
      nameless-let-exp)
    (expression
      ("%lexproc" number expression)
      nameless-proc-exp)

    ))

(sllgen:make-define-datatypes the-lexical-spec the-grammar)

(define show-the-datatypes
  (lambda () (sllgen:list-define-datatypes the-lexical-spec the-grammar)))

(define scan&parse
  (sllgen:make-string-parser the-lexical-spec the-grammar))

(define just-scan
  (sllgen:make-string-scanner the-lexical-spec the-grammar))

;;;;;;;;;;;;;;;; lexical address calculator ;;;;;;;;;;;;;;;;

;;; senv = (list (list symbol))

(define translation-of-program
  (lambda (pgm)
    (cases program pgm
      (a-program (body)
        (a-program                      ; produce a program!
          (translation-of-expression body (init-senv)))))))

(define translation-of-expression 
  (lambda (exp senv)
    (cases expression exp
      (lit-exp (num) exp)
      (var-exp (id) (apply-senv senv id))  ;; produces (lexvar-exp d p)
      (primapp-exp (prim rands)
        (let ((translated-rands
                (translation-of-expressions rands senv)))
          (primapp-exp prim translated-rands)))
      (if-exp (test-exp true-exp false-exp) 
        (if-exp
          (translation-of-expression test-exp senv)
          (translation-of-expression true-exp senv)
          (translation-of-expression false-exp senv)))
      (let-exp (ids rhss body)  
        (nameless-let-exp
          (translation-of-expressions rhss senv)            
          (translation-of-expression body
            (extend-senv ids senv))))
      (proc-exp (ids body)
        (nameless-proc-exp
          (length ids)
          (translation-of-expression body (extend-senv ids senv))))
      (app-exp (rator rands)
        (app-exp
          (translation-of-expression  rator senv)
          (translation-of-expressions rands senv)))
      (else (eopl:error 'translation-of-expression
              "Illegal source expression ~s" exp))
      )))


(define translation-of-expressions
  (lambda (exps env)
    (map
      (lambda (exp) (translation-of-expression exp env))
      exps)))

(define init-senv
  (lambda ()
    '((i v x))))

(define extend-senv
  (lambda (ids senv)
    (cons ids senv)))

(define apply-senv
  (lambda (senv id)
    (let loop ((senv senv) (depth 0))
      (if (null? senv)
        (eopl:error 'apply-senv "unbound variable ~s" id)
        (let ((pos (list-find-position id (car senv))))
          (if (number? pos)
            (lexvar-exp depth pos)
            (loop (cdr senv) (+ depth 1))))))))


;;;;;;;;;;;;;;;; the interpreter ;;;;;;;;;;;;;;;;


(define eval-program 
  (lambda (pgm)
    (cases program pgm
      (a-program (body)
        (eval-expression body (init-nameless-env))))))

;; env is now a nameless-env

(define eval-expression 
  (lambda (exp env)
    (cases expression exp
      (lit-exp (datum) datum)
      (lexvar-exp (depth pos) (apply-nameless-env env depth pos))
      (primapp-exp (prim rands)
        (let ((args (eval-expressions rands env)))
          (apply-primitive prim args)))
      (if-exp (test-exp true-exp false-exp) 
        (if (true-value? (eval-expression test-exp env))
          (eval-expression true-exp env)
          (eval-expression false-exp env)))
      (nameless-let-exp (rhss body) 
        (let ((args (eval-expressions rhss env)))
          (eval-expression body (extend-nameless-env args env))))
      (nameless-proc-exp (nargs body)
        (nameless-closure nargs body env)) 
      (app-exp (rator rands) 
        (let ((proc (eval-expression rator env))
              (args (eval-expressions rands env)))
          (if (procval? proc)
            (apply-procval proc args)
            (eopl:error 'eval-expression
              "Attempt to apply non-procedure ~s" proc))))
      (else
        (eopl:error 'eval-expression
          "Illegal expression in translated code: ~s" exp))
      )))

(define eval-expressions
  (lambda (rands env)
    (map (lambda (x) (eval-expression x env)) rands)))

(define apply-primitive
  (lambda (prim args)
    (cases primitive prim
      (add-prim  () (+ (car args) (cadr args)))
      (subtract-prim () (- (car args) (cadr args)))
      (mult-prim  () (* (car args) (cadr args)))
      (incr-prim  () (+ (car args) 1))
      (decr-prim  () (- (car args) 1))
      (zero-test-prim () (if (zero? (car args)) 1 0))
      )))

(define init-nameless-env 
  (lambda ()
    (extend-nameless-env
;     '(i v x)
      '(1 5 10)
      (empty-nameless-env))))

;;;;;;;;;;;;;;;; booleans ;;;;;;;;;;;;;;;;

(define true-value?
  (lambda (x)
    (not (zero? x))))

;;;;;;;;;;;;;;;; procedures ;;;;;;;;;;;;;;;;

(define-datatype procval procval?
  (nameless-closure 
    (nargs number?)
    (body expression?)
    (env nameless-environment?)))

(define apply-procval
  (lambda (proc args)
    (cases procval proc
      (nameless-closure (nargs body env)
        (eval-expression body (extend-nameless-env args env))))))
               
;;;;;;;;;;;;;;;; nameless environments ;;;;;;;;;;;;;;;;

(define nameless-environment? (list-of vector?))

(define empty-nameless-env
  (lambda ()
    '()))

(define extend-nameless-env
  (lambda (vals env)
    (cons (list->vector vals) env)))

(define apply-nameless-env
  (lambda (env depth pos)
    (cond
      ((null? env) (eopl:error 'apply-nameless-env "No binding!"))
      ((zero? depth) (vector-ref (car env) pos))
      (else (apply-nameless-env (cdr env) (- depth 1) pos)))))

;;;;;;;;;;;;;;;; utilities ;;;;;;;;;;;;;;;;

(define list-find-position
  (lambda (sym los)
    (list-index (lambda (sym1) (eqv? sym1 sym)) los)))

(define list-index
  (lambda (pred ls)
    (cond
      ((null? ls) #f)
      ((pred (car ls)) 0)
      (else (let ((list-index-r (list-index pred (cdr ls))))
              (if (number? list-index-r)
                (+ list-index-r 1)
                #f))))))

(define iota
  (lambda (end)
    (let loop ((next 0))
      (if (>= next end) '()
        (cons next (loop (+ 1 next)))))))

(define difference
  (lambda (set1 set2)
    (cond
      ((null? set1) '())
      ((memv (car set1) set2)
       (difference (cdr set1) set2))
      (else (cons (car set1) (difference (cdr set1) set2))))))

