// Code by Will Guaraldi - guaraldi@ccs.neu.edu
// Version 1.0 11/9/2006
// Created for CSG260 Fall 2006 with Karl Lieberherr
package edu.neu.ccs.satsolver;

import java.util.Iterator;
import java.util.Set;
import java.lang.IllegalArgumentException;

/**
 * Use this class as specified by the interface document.
 */
public class SATSolverUtil {

  private static boolean validR(int r) {
    return r <= 255 && r >= 0;
  }

  private static boolean validFrac(double frac) {
    return frac <= 1.0 && frac >= 0;
  }

  private static double computeMaxBias(Polynomial p) {
    double[] solutions;

    try {
      solutions = p.derivative().solveForZero();

    } catch (Polynomial.PolynomialException pe) {
      System.out.println("Exception with solving " + p + " for zero.");
      return 0.0d;
    }

    double[] x = { 0.0d, 0.0d, 0.0d, 0.0d };

    x[0] = 0.0d; // compute at x=0
    x[1] = 1.0d; // compute at x=1
    if (solutions.length == 1) {
      x[2] = solutions[0];
    } else if (solutions.length == 2) {
      x[2] = solutions[0];
      x[3] = solutions[1];
    }

    double max_result = p.compute(x[0]);
    double max_x = x[0];
    for (int i=1; i < x.length; i++) {
    	if(x[i] >= 0 && x[i] <= 1)
    	{
	      double ret = p.compute(x[i]);
	      if (ret > max_result) {
	        max_result = ret;
	        max_x = x[i];
	      }
    	}
    }

    return max_x;
  }

  /**
   * Calculates the max bias given an InputInitialI instance.
   *
   * Throws IllegalArgumentException if relation number or
   * fraction is out of acceptable ranges.
   */
  public static OutputI calculateBias(InputInitialI input) {
    Set pairs = input.getPairs();
    Iterator iter = pairs.iterator();

    Polynomial final_poly = null;

    while (iter.hasNext()) {
      PairI p = (PairI) iter.next();

      int r = p.getRelationNumber();
      if (!validR(r)) {
        throw new IllegalArgumentException("Relation number is " +
                                           "outside of range.");
      }

      double frac = p.getFraction();
      if (!validFrac(frac)) {
        throw new IllegalArgumentException("Fraction is outside of " +
                                          "range.");
      }

      Polynomial poly = Polynomial.getPolynomial(p.getRelationNumber());
      poly = poly.multiplyByDouble(frac);

      if (final_poly == null) {
        final_poly = poly;
      } else {
        final_poly = final_poly.add(poly);
      }
    }

    double max_bias = computeMaxBias(final_poly);

    return new MaxBiasOutput(final_poly, max_bias);
  }

  /**
   * Calculates the max bias given an InputUpdateI instance.
   *
   * Throws an IllegalArgumentException if the relation number 
   * or fraction are outside of acceptable ranges.
   */
  public static OutputI updateBias(InputUpdateI input) {
    Polynomial final_poly = Polynomial.getPolynomial(input.getPolynomialBefore());

    Set added = input.getAddedPairs();
    Iterator iter = added.iterator();

    while (iter.hasNext()) {
      PairI p = (PairI) iter.next();

      int r = p.getRelationNumber();
      if (!validR(r)) {
        throw new IllegalArgumentException("Relation number is " +
                                           "outside of range.");
      }

      double frac = p.getFraction();
      if (!validFrac(frac)) {
        throw new IllegalArgumentException("Fraction is outside of " +
                                          "range.");
      }

      Polynomial poly = Polynomial.getPolynomial(r);
      poly = poly.multiplyByDouble(frac);
      
      final_poly = final_poly.add(poly);
    }

    Set subtracted = input.getSubtractedPairs();
    iter = subtracted.iterator();

    while (iter.hasNext()) {
      PairI p = (PairI) iter.next();

      int r = p.getRelationNumber();
      if (!validR(r)) {
        throw new IllegalArgumentException("Relation number is " +
                                           "outside of range.");
      }

      double frac = p.getFraction();
      if (!validFrac(frac)) {
        throw new IllegalArgumentException("Fraction is outside of " +
                                          "range.");
      }

      Polynomial poly = Polynomial.getPolynomial(r);
      poly = poly.multiplyByDouble(frac * -1.0);

      final_poly = final_poly.add(poly);
    }

    double max_bias = computeMaxBias(final_poly);

    return new MaxBiasOutput(final_poly, max_bias);
  }
}
