/**
 * Dictates the behavoir of the toolbar button, provides functionality for
 * semantic & syntactic checking, and activates trace operations.
 * 
 * @author JSeifried
 * @version 0.5.3
 */
package jseifried.xaspecteditor.checker;

import jseifried.xaspecteditor.XEditor;
import jseifried.xaspecteditor.dj.XAspectLanguage;
import jseifried.xaspecteditor.view.XView;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkbenchWindowActionDelegate;

public class XAction implements IWorkbenchWindowActionDelegate {

 /** The current workbench window */
 private IWorkbenchWindow window;

 public XAction() {}
 
 public void selectionChanged(IAction action, ISelection selection) {}

 public void dispose() {}

 public void init(IWorkbenchWindow window) {
  this.window = window;
 }

 /**
  * Capture and validate XEditor contents, showing any errors via dialog box.
  * 
  * @param action the current action class object that has been instantiated.
  * @see jseifried.xaspecteditor.XEditor
  * @see jseifried.xaspecteditor.view.XView
  */
 public void run(IAction action)  {
  IWorkbenchPage pages[] = window.getPages();

  //If there are no editors open, show error and exit.
  if(pages[0].getEditors().length == 0) {
   showError(1);
   return;
  }

  //Capture XView object. If none availible, show error and exit.
  XView xview = (XView)pages[0].findView("jseifried.xaspecteditor.xview");
  if(xview==null) {
   showError(0); 
   return;   
  }

  //Capture XView object. If none availible or none in focus, 
  //show error and exit.
  Class activeclass = pages[0].getActiveEditor().getClass();
  if(activeclass != (new XEditor()).getClass()) { 
   showError(1);
   return;
  }

  //Capture the XEditor object and process
  XEditor xeditor = (XEditor)pages[0].getActiveEditor(); 
  pages[0].saveAllEditors(false);
  IDocument doc = xeditor.getDocumentProvider().getDocument(xeditor.getEditorInput());  
  String errors = process(doc);
  int red = 0,green = 0, blue = 0;

   //If semantic or syntactic errors found, color the text red
   if(errors.length()!=0) { 
	red=200; 
   }
   
   // If no errors found, color the text blue and show appropriate message
   else {
    errors = "Aspect structure parsed successfully.\n\nSelector tracing activated...";
	xeditor.activateTrace();
    blue = 200;
   }

   xview.refresh(errors, red, green, blue);
   xeditor.setXView(xview);
   return;
 }
 
 /**
  * Processing of the aspect structure and return of any errors.
  * 
  * @param doc the contents of the editor window.
  * @return string conataining any semantic or syntactic errors.
  * @see jseifried.xaspecteditor.dj.XAspectLanguage
  */
 String process(IDocument doc) {
  try {
   return XAspectLanguage.validate(doc.get());
  }
  //catch errors parsing the contents of the editor, show error and exit.
  catch (Exception e) {
   MessageDialog.openError(window.getShell(), "XAspect Parse Error!", 
     e.getMessage());
   return null;
  }
 }
	
 /**
  * Show appropriate error message to user in a dialog box.
  * 
  * @param errorNum An integer value denoting the error type
  */
 void showError(int errorNum) {
  switch (errorNum) {
   case 0 :
    MessageDialog.openError(window.getShell(), "No XView Availible!", 
      "In order to use this feature an XView window must be open.");
    break;
   case 1 :
    MessageDialog.openError(window.getShell(), "No XEditor Availible!", 
      "In order to use this feature an XEditor must be open and have focus.");
    break;
  }
 }
}