package org.webjinn.cms.model.xml;
import org.webjinn.cms.model.CMSTree;
import org.webjinn.cms.model.Interface;
import org.webjinn.cms.model.Page;
//XML
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Element;
//JDK
import java.io.File;
import java.io.FileOutputStream;
import java.util.List;
import java.util.StringTokenizer;
//From XERCES examples
import dom.Writer;
import org.apache.xerces.parsers.DOMParser;




/** XML-backed implementation of the CMS tree */
public class XMLCMSTree implements CMSTree {

	/** The url of an xml file that stores CMS tree */
	private String fileURI;

	/** DOM representation of the CMS tree */
	private Document doc;

  /** DOM Writer */
	private	Writer writer;

	//The following are auxiliary fields 
	/** XML Element that represents Tree root node */
	private Node rootNode;
	/** XML Element that represents interfaces node */
	private Node interfacesNode;
	/** XML Element that represents pages node */
	private Node pagesNode;

	

	public XMLCMSTree(String fileURI) {
		this.fileURI=fileURI;
		doc = getDOMTree();
		Node rootNode = doc.getDocumentElement();
		Node interfacesNode = Utils.getChildNodeByName(rootNode,XMLConstants.InterfacesTagName);
		Node pagesNode = Utils.getChildNodeByName(rootNode,XMLConstants.PagesTagName);
		writer = new Writer();
	}

	protected void updateFile() {
		try{
			FileOutputStream fout = new FileOutputStream(new File(fileURI));
			try{
				writer.setOutput(fout,null);
				writer.write(doc);
			} finally {fout.close();}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/** Retrieves array of interfaces */	
	public Interface[] getInterfaces() {
		List interfaces = getInterfaceNodes();
		int length = interfaces.size();
		Interface[] result = new Interface[length];
		for (int i=0;i<length;i++)
			result[i] = new XMLInterfaceImpl((Element)interfaces.get(i), this);
		return result;
	}

	/** Retrieves Interface by name or null if not found */
	public Interface getInterface(String name) {
		List interfaces = getInterfaceNodes();
		for (int i=0;i<interfaces.size();i++)
		  if (((Element)interfaces.get(i)).getAttribute("name").equals(name))	
				return new XMLInterfaceImpl((Element)interfaces.get(i), this);
		return null;		
	}


	/** Adds a new interface to the tree.
	 * Interface name provide must be not null, not empty and
	 * unique among interfaces already in the tree 
	 * If interface with the name provided already exists, 
	 * method does nothing */
	public void addInterface(String name, String descr) {
	  if (name==null || name.length()==0 || getInterface(name)!=null) return;	
		Element interfaceEl = 
			XMLInterfaceImpl.createInterface(doc,name,descr);
		interfacesNode.appendChild(interfaceEl);
		updateFile(); //!!! Update !!!
	}

	/** Removes an interface from the tree. 
	 * Do nothing if Interface object provided is not
	 * found in the tree */
	public void removeInterface(Interface in) {
		if (in instanceof XMLInterfaceImpl) {
			Element inEl = ((XMLInterfaceImpl)in).getXMLElement();
			interfacesNode.removeChild(inEl);
			updateFile(); //!!! Update !!!
		}
	}

	/** Given an interface-uri string returns corresponding tree element 
	 * or null if not found 
	 * interface-uri: 
	 *  item: "interface-name/menu-name/menu-name/.../menu-name:item-name"
	 *  menu: "interface-name/menu-name/menu-name/.../menu-name"
	 * FIXME: code is stupid. */
	public InterfaceTreeElement resolveInterfaceURI(String uri) {
		try{
			StringTokenizer st = new StringTokenizer(uri,"/");
		  Interface in = getInterface(st.nextToken());
			Menu menu = in.getRootMenu();
			String section = st.nextToken();
			String menuName = getMenuName(section);
			if (menuName==null) return null;
			String itemName = getItemName(section);
			
			if (!menu.getName().equals(menuName)) return null;
			if (st.hasMoreTokens()) 
				if (itemName!=null) return null;
		  else
			  if (itemName==null) 
					return menu; 
				else 
					return menu.getChildItem(itemName);

      InterfaceTreeElement result = null; 
			while (st.hasMoreTokens()) {
				section = st.nextToken();
				result = getTreeElement(menu,section);
				if (result==null) return null;
				if (st.hasMoreTokens())
					if (result instanceof Menu) 
						menu = (Menu)result; 
					else 
						return null;
			}
			return result;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	

	/** Retrieves array of pages */
	public Page[] getPages() {
		List pages = getPageNodes();
		int length = pages.size();
		Page[] result = new Page[length];
		for (int i=0;i<length;i++)
			result[i] = new XMLPageImpl((Element)pages.get(i), this);
		return result;		
	}

	/** Returns Page object by file URI or null if not found */
	public Page getPage(String fileURI) {
		List pages = getPageNodes();
		for (int i=0;i<pages.size();i++)
		  if (((Element)pages.get(i)).getAttribute("file-uri").equals(fileURI))	
				return new XMLPageImpl((Element)pages.get(i), this);
		return null;				
	}

	/** Adds new page information to the tree. */
	public void addPage(String fileURI, String menuURI) {
	  if (fileURI==null || menuURI==null || 
				getPage(fileURI)!=null) return;	
		Element pageEl = XMLPageImpl.createPage(doc,fileURI,menuURI);
		pagesNode.appendChild(pageEl);
		updateFile(); //!!! Update !!!		
	}

	/** Removes a page from the tree. Do nothing if
	 * this page object is not found in the tree */
	public void removePage(Page page) {
		if (page instanceof XMLPageImpl) {
			Element pageEl = ((XMLPageImpl)page).getXMLElement();
			pagesNode.removeChild(pageEl);
			updateFile(); //!!! Update !!!
		}		
	}

	/** Returns List of Interface nodes */
	private List getInterfaceNodes() {
		return Utils.getChildNodesByName(
				interfacesNode,XMLConstants.InterfaceTagName); 
	}
	
	/** Returns List of Page nodes */
	private List getPageNodes() {
		return Utils.getChildNodesByName(
				pagesNode,XMLConstants.PageTagName); 
	}
	
	
	/** Parses file and returns DOM XML Document */
	private Document getDOMTree() {
		try{
			DOMParser parser = new DOMParser();
			return parser.parse(fileURI);
		} catch (Exception e) {
			return null;
		}
	}

	/** Parses a section of the menu-uri and returns its menu name part.
	 * Returns null if an error occurs */
  private String getMenuName(String section) {
		if (section==null) return null;
		int ind = section.indexOf(":");
		if (ind == 0 || ind==section.length()-1) return null;
	  if (ind < 0) return section;
	  if (ind > 0) return section.substring(0,ind); 	
	}

	/** Parses a section of the menu-uri and returns its item name part.
	 * Returns null if an error occurs or not found */
  private String getItemName(String section) {
		if (section==null) return null;
		int ind = section.indexOf(":");
		if (ind <= 0 || ind==section.length()-1) return null;
	  if (ind > 0) return section.substring(ind+1); 	
	}

	/** Takes host Menu element and searches for its descender
	 * specified by menu-uri section (Menu or Item) */  
	private InterfaceTreeElement getTreeElement(Menu host,String uriSection) {
		String menuName = getMenuName(uriSection);
		String itemName = getItemName(uriSection);
		if (menuName==null) return null;
		Item[] items = host.getChildItems();
		for (int i=0;i<items.length; i++) {
			Menu menu = items[i].getChildMenu();
			if (menu.getName().equals(menuName)) {
				if (itemName==null) 
					return menu;
				else
					return menu.getChildItem(itemName);
			}
		}
		return null;
	}

}
