package peterson;
import java.util.*;
import java.io.*;
import java.awt.*;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import pwbuilder.core.*;

/**
 * The TextResourceManager is an object that allows multiple 
 * TextWriters to write to the same TextResource simultaneously
 * without errors by implementing a request / grant / release
 * protocol (Peterson-like).
 * 
 * The TextWriter will send a request to a manager, notifying it
 * that it wants to write to a given resource.  The 
 * TextResourceManager then replies to the writer with a grant
 * once the resource in question is available.  Once completed with
 * writing, the TextWriter should send a release to the manager,
 * notifying it that the resource is free. 
 *  
 * @author      Brian Knolhoff
 */
public class TextResourceManager implements Serializable, 
  TWListener, StateMachineAvailable, ActionListener, Runnable {

    private transient Vector listns = new Vector();
    // we may only use listeners as valid notifiers, and ignore the others. That would make sense!
    private transient boolean[][] requests; 
    // Neutral, Trying and Critical arrays 
    private transient boolean[] N,T,C;
    // Listener objects 
    private transient TRMListener[] listeners; 
    private transient int lsize;

    public TextResourceManager() {
      listns = new Vector();
      doInit();
    }
    
    /** Initializes the arrays */
    private void doInit() {
      lsize = listns.size();
      requests = new boolean[lsize][lsize];
      N = new boolean[lsize];
      T = new boolean[lsize];
      C = new boolean[lsize];
      for (int i=0;i<lsize;i++) {
        for (int j=0;j<lsize;j++)
          requests[i][j]=true;
        N[i]=true;
        C[i]=false;
        T[i]=false;
      }
      listeners = (TRMListener[])listns.toArray(new TRMListener[0]);
    }

    /** Returns internal position of the component object
      (index in the N,T,C, and requests arrays). 
      Returns -1 if an argument object is not a valid listener */
    private int getInternalPos(Object obj) {
      for(int i=0;i<lsize;i++) {
        Object listener = listeners[i];
        try{
          listener = 
            listener.getClass().getMethod("getTarget",null).invoke(listener, new Object[0]);
        } catch (Exception e) { 
          e.printStackTrace();
        }
        if (obj==listener) return i;
      }
      return -1;
    }

    /** Tests if the critical section is currently occupied */ 
    private boolean isLocked() {
      for(int i=0;i<lsize;i++) 
        if (C[i]) return true;
      return false;
    }

    private transient Thread t = null;

    /** get the state machine for this object */
    public StateMachine getStateMachine(SystemDescr system) {
      return new TRMStateMachine(system,this); 
    }

    public void run() {
      System.out.println(this+":listeners="+lsize);
      while(true) {
        synchronized(this) {
          try {wait(500);} catch(InterruptedException e) {}
          if (!isLocked()) {
            for (int i=0;i<lsize;i++)
              if (canEnterCS(i)) {grantAccess(i);break;}
          }
        }
      }
    }

    /** Tests if a process can be granted the access to the critical section */
    private boolean canEnterCS(int i) {
      if (!T[i]) return false;
      for (int j=0;j<lsize;j++) 
        if (!(N[j] || requests[i][j])) return false;
      return true;
    }

    /** Grants the access to the critical section */
    private void grantAccess(int pos) {
      T[pos] = false;
      C[pos] = true;
      for (int j=0;j<lsize;j++)
        requests[j][pos] = true;
      listeners[pos].receiveTRMEvent(new TRMEvent(this));
    }


    /** Updates the internal arrays according to the event.
      * Ignores the event, if event notifier is not valid 
      * (is not a registered listener of TRM events). */
    public void receiveTWEvent(TWEvent event) {
      //if notifier is not valid (is not a registered listener)
      // then ignore the event!
      int pos = getInternalPos(event.getSource());
      if (pos==-1) return;

      /// to be removed!!!!
      if (event.isRequest()) {
       System.out.println("TRM: request event from "+event.getSource());
      } else {
       System.out.println("TRM: release event from "+event.getSource());
      }
      /// to be removed!!!

      //if notifier is valid then update the arrays
      if (event.isRequest()) {
        synchronized(this) {
          N[pos] = false;
          T[pos] = true;
          for (int j=0;j<lsize;j++) 
            if (j!=pos && T[j]) requests[pos][j] = false;
        }
      } else {
        synchronized(this) {
          C[pos] = false;
          N[pos] = true;
        }
      }
    }

    /** Starts the TRM thread if it is not already started */
    public synchronized void actionPerformed(ActionEvent e) {
      if (t==null || !t.isAlive()) {
        try{
          //doInit();
          t = new Thread(this);
          t.start();
        } catch (Exception exc) {
          exc.printStackTrace();
        }
      }
    }

    //////////////////////////////////  Assembly-Time Functionality      

    /** Registers a TRM event listener with the component.
      Is supposed to be executed at assembly time only.  */
    public synchronized void addTRMListener(TRMListener l) {
      listns.addElement(l);    
      doInit();
    }

    /** Unregisters a TRM event listener from the component.
      Is supposed to be executed at assembly time only.  */
    public synchronized void removeTRMListener(TRMListener l) {
      listns.removeElement(l);    
      doInit();
    } 


    private void writeObject(ObjectOutputStream outStream) throws IOException {
      Vector listenersClone;
      synchronized(this) {
        outStream.defaultWriteObject();
        listenersClone = (Vector)(listns.clone());
      }

      //serialize all serializable elements in the vector supplied.
      Iterator listenerIterator = listenersClone.iterator();
      while(listenerIterator.hasNext()) {
        Object aListener = listenerIterator.next();
        if(aListener instanceof Serializable) outStream.writeObject(aListener);
      }
    }

    private void readObject(ObjectInputStream inStream) throws ClassNotFoundException, IOException {
      synchronized(this) {
        inStream.defaultReadObject();
        if(listns == null) listns = new Vector();
      }

      Object listener;
      while(inStream.available() > 0) {
        listener = inStream.readObject();
        if(listener instanceof TRMListener)
          addTRMListener((TRMListener)listener);
      }
    }
  }
