function [P,name] = evaluatefeatures_MSSE(data_win, fs, windowL, time)
%
% Evaluate features for activity classification or sensor placement site 
% recognition.
% 
% Input arguments are:
% - "data_win" it is a vector of samples acquired in "windowL" milliseconds. 
%   data_win is not simply the 3d raw data but
%   it is the norm of acceleration recordings:
%   data = sqrt(ax.^2+ay.^2+az.^2); 
%   Acceleration data must be expressed in 'g' (not as integer values).
%   Before evaluating features a lowpass filter may be applied (e.g 20Hz,
%   order4). A single or multiple data windows can be provided in data_win.
%   If all windows has the same number of elements it is possible to use a
%   DxW matrix (D elements in each W windows). If they are not, 
%   provide a cell array of W elements. Each cell element will the contain
%   a vector D(window)x1 . 
% - "fs" is the sampling frequency
% - "windowL" is the window acquisition time. This is used to estimate the
%   expected number of samples in each window.
% 
% Output arguments are:
% - "P" which contains the features
% - "name" with an identifier for each feature.
% 
% [Andrea Mannini: a.mannini@sssup.it; Created: 2012. Last changes: June 2014]



%% Step 1: evaluate if I am processing a single window or a cell aarray of multiple windows 
% each window has to be interpolated for frequency domain features given
% that it may be that the sampling frequency is not absolutely stable
if iscell(data_win)
%     data_i = data_win;
    W = length(data_win);
    
    for ww = 1:W
        Dati2{ww} = (data_win{ww}); 
        Dati{ww} = data_win{ww} - mean(data_win{ww});
               
        %% assuming that the sampling rate is not absolutely fixed, frequency domain
        % evaluations are done by means of an interpolated version of the signal
        % itself:
        
        tm = (time{ww} - time{ww}(1) + 1/fs) ./10^3;
        %     tm = ((time{ww} - time{ww}(1)) ./10^3) + 1/fs; % minor bug fix, commented because the results in the paper have it.
        ti = 1/fs:1/fs:(windowL/10^3);
        Data_i2{ww} = interp1(tm, Dati{ww}, ti)';
        Data_i2{ww} = Data_i2{ww}(~isnan(Data_i2{ww}));
        Data_i{ww} = Data_i2{ww} - mean(Data_i2{ww});
        
    end
    
else
    [dim, W] = size(data_win);
    
    % if it is not a cell array it means that all windows already have
    % windowL samples. Then there is no need of interpolation.
    for ww = 1:W
        Dati2{ww} = (data_win(:,ww));
        Dati{ww} = Dati2{ww} - mean(Dati2{ww});

        Data_i2{ww} = Dati2{ww}(~isnan(Dati2{ww}));
        Data_i{ww} = Dati{ww}
    end
    
    
    
end

%% Step 2: evaluate features
% fs = 40;
% f_psd = 0:fs/512:fs/2;
f_psd = 0.3:0.1:15;

means_m = zeros(1,W); stds_m = zeros(1,W);
vars_m = zeros(1,W); maxs_m = zeros(1,W); mins_m = zeros(1,W);
p1 = zeros(1,W); p2 = zeros(1,W); f1 = zeros(1,W); f2 = zeros(1,W); ft = zeros(1,W);
pT = zeros(1,W); p1_pT_ratio = zeros(1,W); p1B = zeros(1,W); f1B = zeros(1,W);
wf1 = zeros(1,W); wf2 = zeros(1,W);
Pxx = zeros(length(f_psd ),W);
LowP = zeros(1,W); HiP = zeros(1,W); Hz2P = zeros(1,W);
LowP2 = zeros(1,W); HiP2 = zeros(1,W); Hz2P2 = zeros(1,W);
onset1 = zeros(1,W); onset2 = zeros(1,W); onset3 = zeros(1,W); onset4 = zeros(1,W);
for ww = 1:W
    
    dati2 = Dati2{ww};
    dati = Dati{ww};
    data_i = Data_i{ww};
    data_i2 = Data_i2{ww};

    
    %% easy features
    means_m(:,ww) = squeeze(nanmean(dati2,1));
    stds_m(:,ww)  = squeeze(nanstd(dati2,1));
    vars_m(:,ww)  = squeeze(nanvar(dati2,1));
    
    if ((size(dati2,1)<2) || isempty(dati))
        maxs_m(:,ww)  = nan;
        mins_m(:,ww)  = nan;
        Pxx(:,ww) = nan.*(ones(size(f_psd)));
        p1(ww) = nan;
        p2(ww) = nan;
        f1(ww) = nan;
        f2(ww) = nan;
        ft(ww) = nan;
        pT(ww) = nan;
        p1_pT_ratio(ww) = nan;
        p1B(ww) = nan;
        f1B(ww) = nan;
        LowP(ww)  = nan; HiP(ww)  = nan; Hz2P(ww) = nan;
        LowP2(ww)  = nan; HiP2(ww)  = nan; Hz2P2(ww) = nan;
        onset1(ww)  = nan;onset2(ww)  = nan;onset3(ww)  = nan; onset4(ww)  = nan;
    else
        maxs_m(:,ww)  = max(dati2);
        mins_m(:,ww)  = min(dati2);
       
        
        %% Frequency domain features (Zhang et al.)
        [S,F,T,Pxx(:,ww)] = spectrogram(data_i,length(data_i),0,f_psd,fs);
%         [S,F,T,Pxx(:,ww)] = spectrogram(dati,length(dati),0,512,fs);
        dP = diff(Pxx(:,ww) );
        
% OLD APPROACH (paper version) % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 
                try
                    zc = fzero_data(1:length(dP),dP,0);
                catch
                    disp(['No zero crossing found on derivative, min(abs(.)) considered instead'])
                    [dummy zc] = min(abs(dP));
                end
                zc = round(zc);

                if sum(Pxx(zc(1:2:end),ww))>sum(Pxx(zc(2:2:end),ww)) % identifico i max della Pxx
                    zc(2:2:end) = [];
                else
                    zc(1:2:end) = [];
                end

                dom_f = [F(zc) Pxx(zc,ww)];
                dom_f(F(zc)<0.3,:) = [];  % ignore low frequency components
                
                dom_f = sortrows(dom_f,-2);
                
                p1(ww) = dom_f(1,2);
                f1(ww) = dom_f(1,1);
                try
                    p2(ww) = dom_f(2,2);
                    f2(ww) = dom_f(2,1);
                catch
                    disp([' No second dominant found'])
                    p2(ww) = Pxx(end,ww);
                    f2(ww) = F(end);
                end
                pT(ww) = sum(Pxx(:,ww));
                p1_pT_ratio(ww) = p1(ww)/pT(ww);

                condit = ( dom_f(:,1)>=0.6 & dom_f(:,1)<=2.6);
                dom_fB = dom_f(condit,:);

                if isempty(dom_fB)
                    condit2 = ( F>=0.6 & F<=2.6);
                    [p1B(ww) indF ] = max(Pxx(condit2,ww));
                    ff = F(condit2);
                    f1B(ww) = ff(indF);
                else
                    p1B(ww) = dom_fB(1,2);
                    f1B(ww) = dom_fB(1,1);
                end
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 

 
% % New approach, commented.  % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 
%         [pks,locs] = findpeaks(Pxx(:,ww),'SORTSTR','descend');
%         ind_allowed = find (F>0.3,1,'first');
%         pks(locs<ind_allowed)= [];
%         locs(locs<ind_allowed)= [];
%         
%         p1(ww) = pks(1);
%         f1(ww) = F(locs(1));
%         p2(ww) = pks(2);
%         f2(ww) = F(locs(2));
%         
%         pT(ww) = sum(Pxx(:,ww));
%         p1_pT_ratio(ww) = p1(ww)/pT(ww);        
%         
%         condit = F(locs)>=0.6 & F(locs)<=2.6;
%         pks2 = pks(condit);
%         locs2 = locs(condit);;
%         
%         try
%             f1B(ww) = F(locs2(1));
%             p1B(ww) = pks2(1);
%         catch
%             condit2 = ( F>=0.6 & F<=2.6);
%             [p1B(ww) indF ] = max(Pxx(condit2,ww));
%             ff = F(condit2);
%             f1B(ww) = ff(indF);            
%         end
%         
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 

        if  ( ww>1 && ~isnan(f1(ww-1)))
            ft(ww) = f1(ww)/f1(ww-1);
        else
            ft(ww) = eps;
        end
        
        %% wavelet
        Nwlt = 8;
%         data_i2(isnan(data_i2))=[];
        [C,L] = wavedec(data_i2,Nwlt,'db10');
        
        clear det appr
        % %         figure(88);
        for i = 1:Nwlt
            det(i,:) = wrcoef('d',C,L,'db10',i);
            det_2(i) = det(i,:)*det(i,:)';
            % %             subplot(10,1,Nwlt-i+3), plot(det(i,:),'.-'), axis tight;
        end
        appr = wrcoef('a',C,L,'db10',Nwlt);
        
        det56_2 = det(5,:)* det(5,:)' + det(6,:)* det(6,:)';
        
        wf1(ww) = det56_2 / ( (data_i2')*(data_i2')' );
        wf2(ww) = det56_2 /sum(det_2);
        
        
        
    end
end

P = [means_m; stds_m; vars_m; maxs_m; mins_m; maxs_m-mins_m; p1; p2; f1; f2; pT; p1_pT_ratio; p1B; f1B; ft; wf1; wf2;];
name = {'means_m'; 'stds_m'; 'vars_m'; 'maxs_m'; 'mins_m'; 'maxs_m-mins_m'; 'p1'; 'p2'; 'f1'; 'f2'; 'pT'; 'p1_pT_ratio'; 'p1B'; 'f1B'; 'ft'; 'wf1'; 'wf2';}; 
