# Provides:  MasterSlave(), IsUpToDate(), MStrace (default value is true)
# Requires user functions:  SetTaskInput(), DoTask(), GetTaskOutput(),
#				UpdateEnvironment()
# Requires from slavelist.g: SendMsg(), RecvMsg(), BroadcastMsg(),
#         ReadEvalFromString(), SENDRECV_TAG, MASTER_SLAVE_QUIT_TAG,
#         BROADCAST_TAG, MASTER_SLAVE_PING_TAG, PING_TAG
Revision.masslave_g := "@(#)$Id: masslave.g,v 0.1 $";

# External functions:
DeclareGlobalFunction("MasterSlave");
DeclareGlobalFunction("IsUpToDate");
DeclareGlobalFunction("CONTINUATION_ACTION");
DeclareGlobalFunction("CONTINUATION");
MStrace := true;
# Variables set to statistics on master and slaves
MStime := 0; MSnrTasks := 0; MSnrUpdates := 0; MSnrRedos := 0;

# Internal functions;  Don't call them.  GAP needs these declarations.
DeclareGlobalFunction("TOPCmaster");
DeclareGlobalFunction("TOPCslave");
MScontinuation := 0;
lastUpdateTag := -1;
DeclareGlobalFunction("NextTag");
DeclareGlobalFunction("TOPCmasterInit");
DeclareGlobalFunction("TOPCslaveInit");
DeclareGlobalFunction("TOPCmasterSendTaskInput");
DeclareGlobalFunction("TOPCmasterRecvTaskOutput");
DeclareGlobalFunction("InitSlaveArray");
DeclareGlobalFunction("IsSlaveArrayEmpty");
DeclareGlobalFunction("IsSlaveArrayFull");
# Probably records make more sense for this part.
DeclareGlobalFunction("MakeSlaveArraySlot"); #w/ input, tag
DeclareGlobalFunction("ExistsSlaveArraySlot");
DeclareGlobalFunction("RemoveSlaveArraySlot");
DeclareGlobalFunction("SlaveArrayTag");
DeclareGlobalFunction("SlaveArrayInput");
DeclareGlobalFunction("SlaveArrayOutput");
DeclareGlobalFunction("SetSlaveArrayOutput");

#============================================================
#Example:

NO_ACTION :=-1; NOTASK :=-1; # dummy values to avoid "Unbound global var error"
DeclareGlobalFunction("MSList");
MakeReadWriteGVar("MSList");
MSexample := function()
  Print("Defining new function using MasterSlave():   MSList(list, fnc)\n");
  MSList := function(list, fnc)
    local i, result, SetTaskInput, DoTask, GetTaskOutput, UpdateEnvironment;
    result := [];
    i := 0;
    SetTaskInput := function()
      if i < Length(list) then i := i+1; return i;
      else return NOTASK;
      fi;
    end;
    DoTask := i -> fnc(list[i]);   # Apply fnc above.
    GetTaskOutput := function(output, input)
      result[input] := output;
      return NO_ACTION;
    end;
    UpdateEnvironment := function(output, input); end;
    MasterSlave( SetTaskInput, DoTask, GetTaskOutput, UpdateEnvironment );
    return result;
  end;
  Print("Now you can execute, for example:\n",
        "     BroadcastMsg( PrintToString(\"MSList := \", MSList) );\n",
        "     ParEval( \"MSList( [10..20], x->x^2 )\" );\n",
        "Or else you can define and then execute:\n",
        "     ParList := function(list,fnc)\n",
        "       return ParCallFuncList( MSList, [list, fnc] );\n",
        "     end;\n",
        "     ParList([10..20],x->x^2);\n",
        "(Note the current value of:  MStrace := ", MStrace, "; )\n",
        "You can also examine the code of MSList() by typing Print(MSList);\n" );
end;

#============================================================
# Primary External Functions

#enumerate constants
# Can these be made immutable?  Maybe using AsList(...)
NO_ACTION := ["NO_ACTION"];
REDO := ["REDO_ACTION"]; UPDATE := ["UPDATE_ACTION"];
 CONTINUE := ["CONTINUATION_ACTION()"];
REDO_ACTION := REDO; UPDATE_ACTION := UPDATE;
# NOTASK must be unique GAP object as tested by IsIdenticalObj()
NOTASK := ["NOTASK"];

InstallGlobalFunction(MasterSlave,
 function( SetTaskInput, DoTask, GetTaskOutput, UpdateEnvironment)
  local time;
  time := Runtime(); MSnrTasks := 0; MSnrUpdates := 0; MSnrRedos := 0;
  if MPI_Comm_rank() = 0 then
    TOPCmaster(SetTaskInput, GetTaskOutput, UpdateEnvironment);
  else
    TOPCslave(DoTask, UpdateEnvironment);
  fi;
  MStime := Runtime() - time;
end);

InstallGlobalFunction(IsUpToDate, function()
  return SlaveArrayTag( MPI_Get_source() ) > lastUpdateTag
        # or tag marker turned over
        or lastUpdateTag - SlaveArrayTag( MPI_Get_source() ) >
           SENDRECV_TAG / 2;;
end);

InstallGlobalFunction( CONTINUATION, function( taskInput )
  MScontinuation := taskInput;
  return CONTINUE;
end);
# alias for CONTINUATION
InstallGlobalFunction( CONTINUATION_ACTION, CONTINUATION);

#============================================================
#Internal functions for MasterSlave

InstallGlobalFunction( TOPCmaster,
 function( SetTaskInput, GetTaskOutput, UpdateEnvironment )
  local lastInput, slave;

  TOPCmasterInit();

  while true do
    # First priority is to update environment (minimize task dependencies)
    while MPI_Iprobe() do
      TOPCmasterRecvTaskOutput( GetTaskOutput, UpdateEnvironment );
    od;

    # Now generate new tasks for empty slots
    lastInput := TOPCmasterSendTaskInput( SetTaskInput );

    # See if we're done, and break out of while loop, if so.
    if IsIdenticalObj(lastInput, NOTASK) and IsSlaveArrayEmpty() then
      # Incrementing currTag() is harmless, since we're leaving
      if NextTag() = 1 then
        Print("Warning:  First task of MasterSlave() was NOTASK.\n");
      fi;
      for slave in [1..MPI_Comm_size()-1] do
        SendMsg( false, slave, MASTER_SLAVE_QUIT_TAG );
      od;
      break;
    fi;

    # IsSlaveArrayEmpty() is false and TaskInput's already sent;
    #   wait to receive TaskOutput
    TOPCmasterRecvTaskOutput( GetTaskOutput, UpdateEnvironment );
  od;
end);

InstallGlobalFunction( TOPCslave, function( DoTask, UpdateEnvironment )
  local msg;

  TOPCslaveInit();

  while true do
    msg := RecvMsg();
    if MPI_Get_tag() = BROADCAST_TAG then # then an update
      MSnrUpdates := MSnrUpdates + 1;
      UpdateEnvironment( msg[1], msg[2] );
    elif MPI_Get_tag() = MASTER_SLAVE_QUIT_TAG or MPI_Get_tag() = QUIT_TAG then
      break;
    elif MPI_Get_tag() = MASTER_SLAVE_PING_TAG then
      SendMsg( false, false, MASTER_SLAVE_PING_TAG );
    else
      MSnrTasks := MSnrTasks + 1;  # This includes new task and REDO
      SendMsg( DoTask( msg ), 0);  # destination 0 is master
    fi;
  od;
end);

InstallGlobalFunction(TOPCmasterSendTaskInput, function( SetTaskInput )
  local slave, input;
  for slave in [1..MPI_Comm_size()-1] do
    if not ExistsSlaveArraySlot(slave) then
      input := SetTaskInput();
      if IsIdenticalObj(input, NOTASK) then break; fi; # leave for loop
      if ForAny([NO_ACTION,REDO,UPDATE,CONTINUE], x->IsIdenticalObj(input,x))
        then Error("SetTaskInput() returned the action, ", input[1],
                   ", instead of NOTASK or task.");
      fi;
      MSnrTasks := MSnrTasks + 1;
      MakeSlaveArraySlot( slave, input );
      if MStrace then
        Print("master -> ", slave, ": "); ViewObj(input); Print("\n");
      fi;
      SendMsg( input, slave, SlaveArrayTag(slave) ); 
    fi;
  od;
  # If no slave available, input undefined.  Then it was an error to call this.
  return input;
end);

InstallGlobalFunction(TOPCmasterRecvTaskOutput,
 function( GetTaskOutput, UpdateEnvironment )
  local output, action, slave;
  output := RecvMsg();
  slave := MPI_Get_source();
  if MStrace then
    Print(slave, " -> master: "); ViewObj(output); Print("\n");
  fi;
  SetSlaveArrayOutput( slave, output );
  action := GetTaskOutput( output, SlaveArrayInput( slave ) );
  if IsIdenticalObj( action, NO_ACTION ) then
    RemoveSlaveArraySlot( slave );
  elif IsIdenticalObj( action, REDO ) then
    if MStrace then
      Print("REDO on slave ", slave, "\n");
    fi;
    MSnrRedos := MSnrRedos + 1;
    SendMsg( SlaveArrayInput(slave), slave, SlaveArrayTag(slave) );
  elif IsIdenticalObj( action, UPDATE ) then
    if MStrace then
      Print("UPDATE: "); ViewObj(output); Print("\n");
    fi;
    MSnrUpdates := MSnrUpdates + 1;
    # Slave seeing broadcast tag will assume UpdateEnvironment()
    BroadcastMsg( [output, SlaveArrayInput( slave )] );
    UpdateEnvironment( output, SlaveArrayInput( slave ) );
    RemoveSlaveArraySlot( slave );
  elif IsIdenticalObj( action, CONTINUE ) then
    # GetTaskOutput() must have returned CONTINUATION(), which
    #   sets MScontinuation, and then returns CONTINUE
    SendMsg( MScontinuation, slave, SlaveArrayTag(slave) );
  else Error("MasterSlave:  GetTaskOutput returned invalid action: ", action);
  fi;
end);

#============================================================
#Initialize master and slave before doing work

# Ping all slaves: see if they're alive, in MasterSlave mode; InitSlaveArray()
InstallGlobalFunction(TOPCmasterInit, function()
  local slave;
  for slave in [1..MPI_Comm_size()-1] do
    SendMsg( false, slave, PING_TAG );
    SendMsg( false, slave, MASTER_SLAVE_PING_TAG );
  od;
  for slave in [1..MPI_Comm_size()-1] do
    MPI_Probe(slave);   # We'll get some reply, unless slave is totally sick.
    if MPI_Get_tag() = PING_TAG then
      RecvMsg(slave);
      Error("Slave ", MPI_Get_source(), " not in MasterSlave mode.\n\n");
    elif MPI_Get_tag() = MASTER_SLAVE_PING_TAG then
      RecvMsg(slave);
    else Error("MasterSlave() begun while messages for",
               "  master are pending.\n",
               "You can type:  FlushAllMsgs()  and re-start.\n\n");
      # Ideally:  MPI_Irecv("",slave,PING_TAG);
      #   MPI_Irecv("",slave,MASTER_SLAVE_PING_TAG);
      #   in this case, but MPINU not defined well enough to handle that.
      # Does the MPI standard call that an error?
    fi;
  od;
  InitSlaveArray();
end);

# Look for and reply to initial ping from master
InstallGlobalFunction(TOPCslaveInit, function()
  RecvMsg( 0 );
  if MPI_Get_tag() <> PING_TAG then
    Error("Expected ping: rank ", MPI_Comm_rank() );
  fi;
  RecvMsg( 0 );  # MASTER_SLAVE_PING_TAG, in case last was a coincidence.
  if MPI_Get_tag() = MASTER_SLAVE_PING_TAG then
    SendMsg( false, 0, MASTER_SLAVE_PING_TAG );
  else
    Error("Did not receive MasterSlave ping: rank ", MPI_Comm_rank() );
  fi;
end);

#============================================================
# Data Structures and Utils for record keeping about slaves and pending tasks

currTag := 0;
lastUpdateTag := -1;
slaveArray := [];
numSlaveArraySlots := 0;

InstallGlobalFunction(NextTag, function()
   currTag := currTag+1;
   if currTag >= SENDRECV_TAG then currTag := 1; fi;
   return currTag;
end);
InstallGlobalFunction(InitSlaveArray, function()
  numSlaveArraySlots := 0;
  slaveArray := [];
  currTag := 0;
  lastUpdateTag := -1; # Initially guarantee:  IsUpToDate() = true
end);

# Hide slaveArray data structure
# Could use records:
#      rec(tag:=NextTag(),input:=taskInput,output:=taskOutput)
# and   slaveArray[slave].tag, etc.
# but maybe    SlaveArrayTag(slave)   is just as clear?
InstallGlobalFunction(IsSlaveArrayEmpty, function()
  return numSlaveArraySlots = 0; end);
InstallGlobalFunction(IsSlaveArrayFull, function()
   return ( numSlaveArraySlots = MPI_Comm_size() - 1 );
end);
InstallGlobalFunction(SlaveArrayTag, slave -> slaveArray[slave][1]);
InstallGlobalFunction(SlaveArrayInput, slave -> slaveArray[slave][2]);
InstallGlobalFunction(SlaveArrayOutput, slave -> slaveArray[slave][3]);
InstallGlobalFunction(SetSlaveArrayOutput, function( slave, output )
   slaveArray[slave][3] := output;
end);
InstallGlobalFunction(MakeSlaveArraySlot, function( slave, task )
  numSlaveArraySlots := numSlaveArraySlots + 1;
  slaveArray[slave] := [NextTag(), task];
end);
InstallGlobalFunction(ExistsSlaveArraySlot,
  slave -> IsBound(slaveArray[slave]));
InstallGlobalFunction(RemoveSlaveArraySlot, function( slave )
  numSlaveArraySlots := numSlaveArraySlots - 1;
  Unbind(slaveArray[slave]);
end);

#============================================================
# Sequential MasterSlave

SeqMasterSlave :=
  function(SetTaskInput, DoTask, GetTaskOutput, UpdateEnvironment)
  local taskInput, taskOutput, action;
  InitSlaveArray();
  while true do
    taskInput := SetTaskInput();
    if taskInput = NOTASK then break; fi;
    repeat
      taskOutput := DoTask( taskInput );
      action := GetTaskOutput( taskOutput, taskInput );
      if action = CONTINUE then taskInput := MScontinuation; fi;
    until action = NO_ACTION or action = UPDATE;
    if action = UPDATE then
      UpdateEnvironment( taskOutput, taskInput );
    fi;
  od;
end;


#============================================================
# Experiment with RawMasterSlave
# The idea is that a traditional SetTaskInput() is a kind of GAP iterator.
# However, sometimes the original sequential code produces task inputs
#  inside of complicated nested loops.  In such cases, it is difficult
#  to create a corresponding iterator (and co-routines or threads would
#  in fact be the ideal language construct).
#  To get around this, we replace a single call to MasterSlave() by:
# BeginRawMasterSlave( DoTask, GetTaskOutput, UpdateEnvironment )
# RawSetTaskInput( taskInput )
# EndRawMasterSlave()
# The application can then call RawSetTaskInput() repeatedly with the
#   new task inputs, before completing the computation by a call
#   to EndRawMasterSlave()

# We'd need a stack of these, if we want to allow recursion.
rawGetTaskOutput := ReturnFail; # arb. declaration
rawUpdateEnvironment := Print;  # arb. declaration

BeginRawMasterSlave := function( DoTask, GetTaskOutput, UpdateEnvironment )
  if MPI_Comm_rank() = 0 then
    TOPCmasterInit();
    rawGetTaskOutput := GetTaskOutput;
    rawUpdateEnvironment := UpdateEnvironment;
  else
    TOPCslave(DoTask, UpdateEnvironment);
  fi;
end;

EndRawMasterSlave := function()
  while not IsSlaveArrayEmpty() do
    TOPCmasterRecvTaskOutput( rawGetTaskOutput, rawUpdateEnvironment );
  od;
end;

#This does same job as TOPCmaster(), previously.
RawSetTaskInput := function( taskInput )
  local SetTaskInput;

  if taskInput = NOTASK and currTag = 0 then
    Print("WARNING:  RawSetTaskInput received NOTASK on first call.\n");
  fi;

  SetTaskInput := function()
    local tmp;
    tmp := taskInput;
    taskInput := NO_ACTION;  # With one call, we don't have a second task.
    return tmp;
  end;

  # First priority is to update environment (minimize task dependencies)
  while MPI_Iprobe() or IsSlaveArrayFull() do
    TOPCmasterRecvTaskOutput( rawGetTaskOutput, rawUpdateEnvironment );
  od;

  # Now pass taskInput to an empty slot;
  # It's guaranteed to exist since IsSlaveArrayFull() is false.
  TOPCmasterSendTaskInput( SetTaskInput ); # This is a local SetTaskInput
end;

MSexample2 := function()
  MSList := function(list, fnc)
    local i, result, DoTask, GetTaskOutput;
    result := [];
    DoTask := i -> fnc( list[i] );
    GetTaskOutput := function( output, input )
      result[input] := output;
      return NO_ACTION;
    end;

    BeginRawMasterSlave( DoTask, GetTaskOutput, Print );
    for i in [1..Length(list)] do 
      RawSetTaskInput(i);
    od;
    EndRawMasterSlave();
  end;
end;

#============================================================
# Experiment with futures:
#   This experiment causes futures to be computed on slaves in parallel.
#   Futures are probably most useful in shared memory, or else
#   remote processes will not see updates to the environment.
#   In distributed memory, a future cannot use any information beyond the
#   initial value of the environment.  In TOP-C terms,
#   it is unclear what should be the environment, and how it should
#   be updated on remote slaves.
#     If a future is purely functional, there's no dependency on any
#   environment, and it's all okay.

futureArray := [];
futureCounter := 0;
# A language modification would create a special type of variable, FUTURE;
#   Such variables would be created by MakeFuture() and evaluated
#   (and converted into variables of ordinary type) by EvaluateFuture()
MakeFuture := function(command)
  futureCounter := futureCounter+1;
  RawSetTaskInput( [ futureCounter, command ] );
  return futureCounter;
end;
EvaluateFuture := function(counter)
  local tmp;
  while not IsBound(futureArray[counter]) do
    RawSetTaskInput( NOTASK ); # Force RawMasterSlave() to get TaskOutput
  od;
  tmp := futureArray[counter];
  Unbind(futureArray[counter]);
  return tmp;
end;

FutureDoTask := function(command)
  return ReadEvalFromString(command[2]);
end;
FutureGetTaskOutput := function(output,input)
  futureArray[input[1]] := output;
  return NO_ACTION;
end;
BeginFutures := function()
  BeginRawMasterSlave( FutureDoTask, FutureGetTaskOutput, Print );
end;
EndFutures := function() EndRawMasterSlave(); end;
