package edu.neu.ccs.demeterf.lexer;

import java.io.IOException;
import java.io.InputStream;
import edu.neu.ccs.demeterf.lib.List;

public class Lexer{
    InputStream input;
    
    final int[][] EDGES;
    final int[][] FINAL;
    final String[] NAMES;
    final int EOF, SKIP;
    
    public Lexer(InputStream in, ADFA dfa){
        input = in;
        EDGES = dfa.getEDGES();
        FINAL = dfa.getFINAL();
        NAMES = dfa.getNAMES();
        EOF = dfa.EOF();
        SKIP = dfa.SKIP();
    }

    public class Tok{
        int type;
        String image;
        int line, col;
        public Tok(int t, String i, int l, int c){
            type = t; image = i;
            line = l; col = c;
        }
        public String toString(){
            return ("("+NAMES[type]+",\""+
                    escape(image)+"\","+
                    line+","+col+")");
        }
        public String getImage(){ return image; }
        public boolean isEOF(){ return type == EOF; }
        public boolean isSKIP(){ return type == SKIP; }
    }
    
    
    String saved = "";
    boolean eofHit = false;
    int read() throws IOException{
        if(saved.length() == 0){
            int ch = input.read();
            if(ch < 0){
                ch = 0;
                eofHit = true;
            }
            return ch;
        }
        int ch = saved.charAt(0);
        saved = saved.substring(1);
        return ch;
    }
    
    int lineNum = 0;
    int colNum = 0;
    public Tok nextToken(){
        int state = 1,
            finSt = 0, finIdx = 0,
            finLine = 0, finCol = 0;
        String buff = "";

        
        int line = lineNum,
            col = colNum;
        while(true){
            int ch = 0;
            try{ ch = read(); }
            catch(IOException io){
                eofHit = true;
                ch = 0;
            }

            if(ch == '\n'){
                line++;
                col = 0;
            }else
                col++;

            int next = smallTrans(state, ch);
            //int next = trans(state, ch);
            buff += (char)ch;

            if(next == 0){
                if(finSt == 0){
                    if(eofHit)
                        return new Tok(EOF,"", lineNum, colNum);
                    throw new RuntimeException("\n Lexer Error: Line "+line+", Column "+col+
                                               "\n    Char: \'"+escape((char)ch)+"\' ["+((int)ch)+"]"+
                                               "\n    Buff: \""+escape(buff)+"\""+
                                               "\n   State: #"+state);
                }
                Tok ret = new Tok(FINAL[finSt][1],buff.substring(0,finIdx),lineNum,colNum);
                saved = buff.substring(finIdx);
                lineNum = finLine;
                colNum = finCol;
                return ret;
            }

            /* Good State... */
            if(eofHit){
                if(FINAL[state][0] == 1){
                    saved = "";
                    lineNum = line;
                    colNum = col;
                    return new Tok(FINAL[state][1],buff,lineNum,colNum);
                }
                if(buff.length() == 0)
                    return new Tok(EOF,"", lineNum, colNum);
                throw new RuntimeException("\n Unexpected EOF: Line "+line+", Column "+col+
                                           "\n Buffer is: \""+escape(buff)+"\"");
            }
            state = next;
            if(FINAL[state][0] == 1){
                finSt = state;
                finIdx = buff.length();
                finLine = line;
                finCol = col;
            }
        }
    }
 
    public static String escape(String s){
        char str[] = s.toCharArray();
        StringBuffer ret = new StringBuffer("");
        for(char c:str)ret.append(escape(c));
        return ret.toString();
    }
    public static String escape(char c){
        switch(c){
        case '\n':return "\\n";  case '\t':return "\\t";
        case '\b':return "\\b";  case '\r':return "\\r";
        case '\f':return "\\f";  case '\\':return "\\";
        case '\'':return "\\\'"; case '\"':return "\\\"";
        default: return ""+c;
        }
    }
 
    public int trans(int state, int ch){
        return EDGES[state][ch];
    }
   
    public int smallTrans(int state, int ch){
        int[] table = EDGES[state];
        int i = 0;
        while(i < table.length){
            ch -= table[i];
            if(ch < 0){
                return table[i+1];
            }
            i+=2;
        }
        throw new RuntimeException(" NO TRANSITION!! ["+state+"]["+ch+"]");
    }

    public static void main(String[] args){
        test(new TestLexer(System.in));
    }
    
    public static void test(Lexer lex){
        Tok t;
        List<Tok> ts = List.create();
        do{
            t = lex.nextToken();
            //System.out.println("   "+t);
            if(!t.isSKIP())
                ts = ts.push(t);
        }while(!t.isEOF());
        System.out.println(ts.reverse().toString("\n","   "));
    }
}