/* EdgeBypass.java
 * Bryan Chadwick :: 2007
 * Adds the ability to skip/bypass edges and modify BuiltIns
 *   while Traversing */

package edu.neu.ccs.demeterf.control;

import edu.neu.ccs.demeterf.dispatch.Type;
import edu.neu.ccs.demeterf.lib.List;

import java.lang.reflect.Field;

/** Supports the ability to skip/bypass edges of a structure while Traversing.  Essentially
 *   it is a boolean function object: 
 *   <blockquote>
 *      <tt>boolean skip(Class, String);</tt>
 *   </blockquote>
 *   that allows Per-Traversal control.  See {@link edu.neu.ccs.demeterf.Traversal Traversal}
 *   for how its used in construction, and {@link edu.neu.ccs.demeterf.examples Examples}
 *   for how traversal can be controled to affect computation.
 */
public class EdgeBypass extends MutableControl{
    protected List<Edge> skiplist = List.create();
    
    /** Bypass all the edges given */
    public EdgeBypass(){ this(new Edge[0]); }
    /** Bypass all the edges given */
    public EdgeBypass(Edge ... edges){ skiplist = List.create(edges); }
    /** Bypass all the edges of the given classes */
    public EdgeBypass(Class<?> ... cs){ for(Class<?> c:cs)this.addBypassing(c); }
    
    /** Takes a String in the format of "Class.edge "+ (meaning more than one).  We then attempt
      *   to recreate an EdgeBypass with the given edges bypassed. Static methods within
      *   {@link edu.neu.ccs.demeterf.dispatch.Type} can be used to extend the path that is
      *   used to search for classes. */
    public EdgeBypass(String ... bypass){ skiplist = init("", bypass); }
    public EdgeBypass(String bypass){ skiplist = init("", bypass); }
    public EdgeBypass(String pkg, String bypass){ skiplist = init(pkg+".", bypass); }
    
    protected static List<Edge> init(String pkg, String bypass){ return init(pkg,bypass.split(" ")); }
    protected static List<Edge> init(String pkg, String[] bypass){
        if(pkg.length() > 0 && pkg.charAt(pkg.length()-1) != '.')pkg += ".";
        List<Edge> lst = List.create();
        for(String s:bypass){
            int dot = s.lastIndexOf(".");
            if(dot >= 0)
                lst = lst.push(new Edge(Type.classForName(pkg+s.substring(0,dot)),s.substring(dot+1)));
        }
        return lst;
    }
    
    public MutableControl addBypassing(Class<?> c, String f){ return addBypassing(new Edge(c,f)); }
    public MutableControl addBypassing(Edge e){ skiplist = skiplist.push(e); return this; }
    public MutableControl addBypassing(Class<?> c){ 
        Field fs[] = c.getDeclaredFields();
        for(Field f:fs)
            addBypassing(new Edge(c,f.getName()));
        return this;
    }
    
    public MutableControl removeBypassing(Class<?> c, String f){ return removeBypassing(new Edge(c,f)); }
    public MutableControl removeBypassing(Edge e){ skiplist = skiplist.remove(e); return this; }
    
    
    public EdgeBypass addBuiltIn(Class<?> c){ builtIns.add(c); return this; }
    public EdgeBypass addBuiltIns(Class<?> ... cs){
        for(Class<?> c:cs)builtIns.add(c); return this;    
    }

    public boolean skip(Class<?> c, String f){ return skip(new Edge(c,f)); }
    public boolean skip(Edge e){ return skiplist.contains(e); }
    
    public boolean ignore(Class<?> c, String f){ return false; }
    public boolean ignore(Edge e){ return false; }
}