using System;
using System.Collections.Generic;

using Fields = edu.neu.ccs.demeterf.Fields;

namespace edu.neu.ccs.demeterf.lib{


/** Represents a Set of comparable Elements, implemented with RBTrees. */
public class Set<X> : IEnumerable<X>{
    protected readonly RBTree<Wrap<X>> tree;
    private readonly Comparison<X> comp;
    
    /** Field Class */
    public class treeF : Fields.any{}
    
    /** Function Class to help in the merging of Sets.  Merges same elements calling merge(...). */
    public abstract class Merge{ public abstract X merge(X a, X b); }
    
    /** Construct the <i>empty</i> Set of Comparable elements. */
    private Set() : this(Wrap<X>.CComp,RBTree<Wrap<X>>.create<Wrap<X>>()) { }
    /** Construct the <i>empty</i> Set with a separate Comparator. */
    private Set(Comparison<X> c) : this(c,RBTree<Wrap<X>>.create<Wrap<X>>()) { }
    
    internal class Wrapper : List<X>.Map<Wrap<X>>{
        Comparison<X> c;
        public Wrapper(Comparison<X> cc){ c = cc; }
        public override Wrap<X> map(X x){ return new Wrap<X>(x,c); }
    }
    /** Construct a Set from a List with a separate Comparator.  Duplicates will be ignored. */
    private Set(List<X> l, Comparison<X> c) : this(c,RBTree<Wrap<X>>.create<Wrap<X>>(l.map(new Wrapper(c)))) { }
    
    /** Construct a Set from a List of <tt>IComparable</tt> elements.  Duplicates
     *    will be ignored. Left Public to support Parsing */
    public Set(List<X> l) : this(l,Wrap<X>.CComp) { }
    /** Construct a Set from a tree of Wrapped (Comparable) elements. */
    public Set(RBTree<Wrap<X>> t) : this(Wrap<X>.CComp,t) { }
    
    /** Construct the <i>empty</i> Set of Comparable elements. */
    public static Set<X> create<X>() where X : IComparable<X> { return new Set<X>(); }
    /** Construct the <i>empty</i> Set with a separate Comparator. */
    public static Set<X> create(Comparison<X> c){ return new Set<X>(c,RBTree<Wrap<X>>.create<Wrap<X>>()); }
    /** Construct a Set from a List of <tt>IComparable</tt> elements.  Duplicates will be ignored. */
    public static Set<X> create<X>(List<X> l) where X : IComparable<X> { return new Set<X>(l); }
    /** Construct a Set from a List with a separate Comparator.  Duplicates will be ignored. */
    public static Set<X> create(List<X> l, Comparison<X> c){ return new Set<X>(l,c); }
    /** Construct a Set from a number of <tt>IComparable</tt> elements.  Duplicates will be ignored. */
    public static Set<X> create<X>(params X[] xs) where X : IComparable<X> { return new Set<X>(List<X>.create(xs)); }
    
    
    // Private Creators... reverse the order since the list/tree will be Wrapped elements
    private Set(Comparison<X> c, List<Wrap<X>> l) : this(c,RBTree<Wrap<X>>.create(l)) { }
    private Set(Comparison<X> c, RBTree<Wrap<X>> t){ tree = t; comp = c; }
    private Set<X> make(RBTree<Wrap<X>> t){ return new Set<X>(comp,t); }
    private Set<X> make(List<Wrap<X>> lst){ return new Set<X>(comp,lst); }
    private List<Wrap<X>> toWrapList(){ return tree.toList(); }
    
    /** Is the given element present in this Set? */
    public bool contains(X x){ return tree.contains(new Wrap<X>(x,comp)); }
    /** Is this set <i>empty</i>? */
    public bool isEmpty(){ return tree.isLeaf(); }
    /** Return the nuber of elements in this Set */
    public int size(){ return tree.size(); }
    /** Returns a new Set that also contains the given element. */
    public Set<X> add(X x){ return make(tree.insert(new Wrap<X>(x,comp))); }
    /** Returns a new Set that no longer contains the given element. */
    public Set<X> remove(X x){ return make(tree.remove(new Wrap<X>(x,comp))); }
    /** Is this Set a <i>subset</i> of (or equal to) the given set? */
    public bool subseteq(Set<X> s){ return s.tree.containsAll(this.tree); }
    /** Returns a new Set that is the <i>union</i> of <b>this</b> and the given Set. */
    public Set<X> union(Set<X> s){ return make(tree.insertAll(s.tree)); }
    
    class InTree : List<Wrap<X>>.Pred{
        RBTree<Wrap<X>> t;
        public InTree(RBTree<Wrap<X>> tt){ t = tt; }
        public override bool huh(Wrap<X> x){ return t.contains(x); }
    }
    /** Returns a new Set that is the <i>intersection</i> of <b>this</b> and the given Set. */
    public Set<X> intersect(Set<X> s){
        return make(toWrapList().filter(new InTree(s.tree)));
    }
    /** Returns a new Set that is the <i>difference</i> of <b>this</b> and the given Set. */
    public Set<X> difference(Set<X> s){
        return make(toWrapList().filterout(new InTree(s.tree)));
    }
    
    internal class UnWrapper : List<Wrap<X>>.Map<X>{
        public override X map(Wrap<X> w){ return w.GetX(); }
    }
    /** Returns this Set as a List, in the order defined by the Comparator. */
    public List<X> toList(){ return toWrapList().map(new UnWrapper()); }
    
    /** Returns an iterator for this Set, in the order defined by the Comparator. */
    public IEnumerator<X> GetEnumerator(){ return toList().GetEnumerator(); }
    /** Returns an iterator for this Set, in the order defined by the Comparator. */
    System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator(){ return GetEnumerator(); }
    
    class Merger : List<Wrap<X>>.Fold<RBTree<Wrap<X>>>{
        Merge m;
        Comparison<X> comp;        
        public Merger(Merge mm, Comparison<X> c){ m = mm; comp = c; }
        public override RBTree<Wrap<X>> fold(Wrap<X> w, RBTree<Wrap<X>> t){
            if(!t.contains(w))return t.insert(w);
            return t.replace(new Wrap<X>(m.merge(w.GetX(),t.find(w).GetX()),comp));
        }
    }
    /** Merge two sets with like elements merged by the given function object */
    public Set<X> merge(Set<X> s, Merge m){
        return make(s.toWrapList().fold(new Merger(m, comp), tree));
    }
    /** Merge a single element merged by the given function object */
    public Set<X> merge(X x, Merge m){
        Wrap<X> w = new Wrap<X>(x,comp);
        if(!tree.contains(w))return make(tree.insert(w));
        return make(tree.replace(new Wrap<X>(m.merge(x,tree.find(w).GetX()),comp)));
    }
    
    /** Set equality (s == t) is (s.subset(t) && t.subset(s)). */
    public override bool Equals(Object o){
        if(!(o is Set<X>))return false;
        Set<X> s = (Set<X>)o;
        return (s.size() == this.size() && tree.containsAll(s.tree));
    }
    /** Produce a string representation of this Set. */
    public override String ToString(){ return "{ "+toList().ToString(" ","")+" }"; }
    
    /** Return the combined underlying elements HashCode */
    public override int GetHashCode(){ return tree.GetHashCode(); }
}

}