namespace edu.neu.ccs.demeterf.lib{

using System;
using System.Collections;
using System.Collections.Generic;

/** Represents Lisp stype Lists. List.create(...) should be used to create lists, and
 *    push/pop/top/etc... to pull them apart.  All methods are functional, meaning a
 *    given list is an immutable data type: methods return a new list, they do not
 *    change the old one.  The methods can be used to write beautiful recursive
 *    functions :)
 *    
 *    The inner classes (<tt>Build</tt>, <tt>Comp</tt>, <tt>GComp</tt>, <tt>Fold</tt>,
 *    <tt>Map</tt>, <tt>Pred</tt>, and <tt>Zip</tt>) are <i>Function Classes</i> that
 *    allow you to implement needed functionalities over basic traversals/iteration.
 *    
 *    This implementation has been updated to eliminate stack usage, which allows
 *    you (the programmer) to create HUGE lists without any problems ;)
 *    */
public abstract class List<X> : IEnumerable<X>{
    
    /** Compute a String from a List (Visitor) */
    public abstract class Stringer{ public abstract String toString(X f, List<X> r); }
    /** Select Elements of a List */
    public abstract class Pred{
        /** Is this predicate true for the given X? */
        public abstract bool huh(X x);
        /** Return a Predicate that is the negation of this one */
        public Pred negate(){ return new Negate(this); }
        private class Negate : Pred{
            Pred p; public Negate(Pred pp){ p = pp; }
            public override bool huh(X x){ return !p.huh(x); }
        }
        
    }
    /** Compare two List Elements of the same type */
    public abstract class Comp : GComp<X>{}
    /** (General) Compare two List Elements of possibly different types 
     *    (true is "LessThan" for <tt>sort()</tt>, but "Equal" for <tt>same(...)</tt>) */
    public abstract class GComp<Y>{
        /** Compare these two Xs, returning true/false */
        public abstract bool comp(X x, Y y);
        private GComp<Y> that;
        
        /** Return a general comparator with flipped comparision (and types) */
        public List<Y>.GComp<X> flip(){ return new Flip(this); }
        private class Flip : List<Y>.GComp<X>{
            List<X>.GComp<Y> par;
            public Flip(List<X>.GComp<Y> p){ par = p; }
            public override bool comp(Y y, X x){ return par.comp(x, y); }
        }
        
        /** Return a Predicate that uses the comparator (later) using the given X as
         *    its first argument */
        public List<X>.Pred curry(Y y){ return new Curry(this,y); }
        /** Return a (reversed) Predicate that uses the comparator (later) using the given Y as
         *    its second argument */
        public List<Y>.Pred revCurry(X x){ return new RCurry(this,x); }

        private class Curry : List<X>.Pred{
            Y y; GComp<Y> c; public Curry(GComp<Y> cc, Y yy){ c = cc; y = yy; }
            public override bool huh(X x){ return c.comp(x, y); }
        }
        private class RCurry : List<Y>.Pred{
            X x; GComp<Y> c; public RCurry(GComp<Y> cc, X xx){ c = cc; x = xx; }
            public override bool huh(Y y){ return c.comp(x, y); }
        }
    }
    /** Apply a function to each element of the list */
    public abstract class Map<Y>{ public abstract Y map(X x); }
    /** Fold the List into a single Value */
    public abstract class Fold<Y>{ public abstract Y fold(X x, Y y); }
    /** Zip two Lists into a single Value */
    public abstract class Zip<Y,Z>{ public abstract Z zip(X x, Y y); }
    /** Build a List from the sequence of integers [0..len-1] */
    public abstract class Build{ public abstract X build(int i); }

    /** Equals Comparator */
    private class EqualComp : Comp{
        public override bool comp(X x, X y){ return x.Equals(y); }
    }
    
    /** Create an Empty List */
    public static List<X> create(){ return new Empty<X>(); }
    /** Create a List from an array/variable arguments */
    public static List<X> create(params X[] xa){ return create(xa,0); }
    /** Create a List from a fixed array, starting at index 'i' */
    public static List<X> create(X[] xa, int i){
        List<X> res = List<X>.create();
        for(;i < xa.Length;i++)res = res.push(xa[i]);
        return res.reverse();
    }
    /** Create a List from an Iterable... */
    public static List<X> create(IEnumerable<X> xs){
        List<X> res = List<X>.create();
        foreach(X x in xs)res = res.push(x);
        return res.reverse();
    }

    private readonly int len;
    /** Default Constructor */
    public List(int l){ len = l; }
    
    /** Push a X on the front of this List */
    public List<X> push(X x){ return new Cons<X>(x,this); }
    /** Push all Elements of the given List on the front of this List */
    public List<X> push(List<X> xs){ return xs.append(this); }
    /** Reverse this List */
    public List<X> reverse(){ return reverse(length()); }
    /** Reverse the first i elements of this List */
    public List<X> reverse(int i){
        List<X> front = List<X>.create(), back = this;
        while(!back.isEmpty() && i-- > 0){
            X x = back.top();
            back = back.pop();
            front = front.push(x);
        }
        return front;
    }
    /** Canonical ToString */
    public override String ToString(){ return ToString(" ",""); }
    /** Return the Index of the given element */
    public int index(X x){ return index(new EqualComp().curry(x)); }
    /** Return the Index of the first match */
    public int index(List<X>.Pred p){
        int i = 0;
        foreach(X x in this){
            if(p.huh(x))return i;
            i++;
        }
        return -1;
    }
    /** Is the given list have the same elements as this list? */
    public bool same(List<X> l){ return containsAll(l) && l.containsAll(this); }
    /** Is the given list have the same elements as this list using the given comparer? */
    public bool same(List<X> l, Comp c){ return sameG(l,c); }
    /** Is the given list have the same elements as this list using the given comparer? */
    public bool sameG<Y>(List<Y> l, GComp<Y> c){ return containsAllG<Y>(l,c) && l.containsAllG<X>(this,c.flip()); }
    /** Return this List without the first k Elements */
    public List<X> pop(int k){ return (k == 0)?this:pop().pop(k-1); }
    
    /** Append another List to the end of this List */
    public virtual List<X> append(List<X> l){
        List<X> rev = reverse();
        return l.revpush(rev);
    }
    /** Append an element to the end of this List */
    public virtual List<X> append(X x){ return append(List<X>.create(x)); }
    /** Return the first Element of this List */
    public abstract X top();
    /** Return this List without the first Element */
    public abstract List<X> pop();
    /** Is this List Empty? */
    public abstract bool isEmpty();
    
    /** Does this Predicate match any element in this List? */
    public bool ormap(List<X>.Pred p){
        foreach(X x in this)
            if(p.huh(x))return true;
        return false;
    }
    /** Does this Predicate match all the elements in this List? */
    public bool andmap(List<X>.Pred p){ return !ormap(p.negate()); }
    
    /** Does the given X occur in this List? */
    public virtual bool contains(X x){ return contains(new EqualComp().curry(x)); }
    /** Does this Predicate match anything in this List? */
    public virtual bool contains(List<X>.Pred p){ return ormap(p); }
    /** Does the given X occur in this List? */
    public bool containsG<Y>(Y y, GComp<Y> c){ return contains(c.curry(y)); }
    
    private class AnyP<Y> : List<X>.Pred{
        public AnyP(List<Y> ll, GComp<Y> cc){ l = ll; c  = cc; }
        List<Y> l; GComp<Y> c;
        public override bool huh(X x){ return l.contains(c.revCurry(x)); }
    }
    /** Does this List contain any of the given List's Elements? */
    public virtual bool containsAny(List<X> l){ return containsAny(l, new EqualComp()); }
    /** Does this List contain any of the given List's Elements using the given comparer? */
    public virtual bool containsAny(List<X> l, Comp c)
    {  return this.ormap(new AnyP<X>(l,c)); }
    /** Does this List contain any of the given List's Elements using the given comparer? */
    public bool containsAnyG<Y>(List<Y> l, GComp<Y> c){
        return this.ormap(new AnyP<Y>(l,c));
    }
    
    /** Does this List contain all of the given List's Elements? */
    public virtual bool containsAll(List<X> l){ return containsAll(l, new EqualComp()); }
    /** Does this List contain all of the given List's Elements using the given comparer? */
    public virtual bool containsAll(List<X> l, Comp c){
        return l.andmap(new AnyP<X>(this,c));
    }
    /** Does this List contain all of the given List's Elements using the given comparer? */
    public bool containsAllG<Y>(List<Y> l, GComp<Y> c){
        return l.andmap(new List<Y>.AnyP<X>(this,c.flip()));
    }
    /** Return the given X, throws a RuntimeException if not there */
    public X find(X x){ return find(new EqualComp().curry(x)); }
    /** Return the first matching X, throws a RuntimeException if not there */
    public X find(List<X>.Pred p){
        foreach(X x in this)
            if(p.huh(x))return x;
        throw new Exception("No Match Found");
    }
    /** Return the given X, throws a RuntimeException if not there */
    public X findG<Y>(Y y, GComp<Y> c){ return find(c.curry(y)); }
    /** Remove the first occurence of the given X */
    public List<X> remove(X x){ return remove(new EqualComp().curry(x)); }
    /** Remove the first occurence of the given X */
    public List<X> removeG<Y>(Y y, GComp<Y> c){ return remove(c.curry(y)); }
    
    /** Push the reverse of the given list on the front of this one */
    private List<X> revpush(List<X> l){
        List<X> ret = this;
        foreach(X x in l)ret = ret.push(x);
        return ret;
    }
    /** Remove the first matching X */
    public List<X> remove(List<X>.Pred p){
        List<X> front = List<X>.create(), back = this;
        while(!back.isEmpty()){
            X x = back.top();
            back = back.pop();
            if(p.huh(x))return back.revpush(front);
            front = front.push(x);
        }
        return front.reverse();
    }
    /** The Length of This List */
    public int length(){ return len; }
    /** Lookup the i^th item in this List */
    public X lookup(int i){
        List<X> l = this; 
        while(i-- > 0)l = l.pop();
        return l.top();
    }
    /** To String, with a seperator and prefix */
    public virtual String ToString(String sep, String pre){
        String ret = "";
        bool first = true;
        foreach(X x in this){
            if(!first)ret += sep;
            else first = false;
            ret += pre+x;
        }
        return ret;
    }
    /** To String using a Stringer (Visitor) */
    public virtual String ToString(Stringer s){
        String ret = "";
        List<X> lst = this;
        while(!lst.isEmpty()){
            ret += s.toString(lst.top(),lst.pop());
            lst = lst.pop();
        }
        return ret;
    }
    /** Filter out all the same Elements */
    public List<X> filterout(X x){ return filterout(new EqualComp().curry(x)); }
    /** Filter out all the matching Elements */
    public List<X> filterout(List<X>.Pred p){
        return filter(p.negate());
    }
    /** Filter out all the non-same Elements */
    public List<X> filter(X x){ return filter(new EqualComp().curry(x)); }
    /** Filter out all the non-matching Elements */
    public virtual List<X> filter(List<X>.Pred p){
        List<X> keep = List<X>.create();
        foreach(X x in this)
            if(p.huh(x))keep = keep.push(x);
        return keep.reverse();
    }
    /** Count the elements that match the given Predicat */
    public int count(List<X>.Pred p){
        int c = 0;
        foreach(X x in this)
            if(p.huh(x))c++;
        return c;
    }
        
    class RMDup : Fold<List<X>>{
        Comp c;
        public RMDup(Comp cc){ c = cc; }
        public override List<X> fold(X x, List<X> l){
            if(l.contains(c.curry(x)))return l;
            return l.push(x);
        }
    }
    /** Remove duplicate items from this list */
    public List<X> removeDuplicates(){ return removeDuplicates(new EqualComp()); }
    /** Remove duplicate items from this list */
    public List<X> removeDuplicates(Comp c){
        return this.fold(new RMDup(c), List<X>.create());
    }
    /** Fold this List to a single Value (Left to Right)*/
    public Y fold<Y>(List<X>.Fold<Y> f, Y b){ return foldl(f,b); }
    /** Fold this List to a single Value (Left to Right)*/
    public virtual Y foldl<Y>(List<X>.Fold<Y> f, Y b){
        foreach(X x in this)b = f.fold(x, b);
        return b;
    }
    /** Fold this List to a single Value (Right to Left)*/
    public virtual Y foldr<Y>(List<X>.Fold<Y> f, Y b){ return reverse().foldl(f, b); }
    /** Apply a function to each Element of this List */
    public List<Y> map<Y>(List<X>.Map<Y> m){
        List<Y> ret = List<Y>.create();
        foreach(X x in reverse())ret = ret.push(m.map(x));
        return ret;
    }
    /** Add an Element to this list at the given index */
    public List<X> add(X a, int i){
        List<X> front = List<X>.create(), back = this;
        while(!back.isEmpty()){
            X x = back.top();
            back = back.pop();
            if(i-- == 0)return back.push(x).push(a).revpush(front);
            front = front.push(x);
        }
        if(i == 0)return front.push(a).reverse();
        throw new Exception("Bad Add");
    }
    /** Remove an Element from this list at the given index */
    public List<X> remove(int i){
        List<X> front = List<X>.create(), back = this;
        while(!back.isEmpty()){
            X x = back.top();
            back = back.pop();
            if(i-- == 0)return back.revpush(front);
            front = front.push(x);
        }
        throw new Exception("Bad Remove");
    }
    /** Insert a number of Elements into this SORTED list */
    public List<X> insert(IEnumerable<X> xs, List<X>.Comp c){
        List<X> ths = this;
        foreach(X x in xs)ths = ths.insert(x,c);
        return ths;
    }
    /** Insert an Element into this SORTED list using the given Comparison */
    public List<X> insert(X a, List<X>.Comp c){
        List<X> front = List<X>.create(), back = this;
        while(!back.isEmpty()){
            X x = back.top();
            if(c.comp(a, x))return back.push(a).revpush(front);
            back = back.pop();
            front = front.push(x);
        }
        return front.push(a).reverse();
    }
    /** Sort this List using the given Comparison */
    public List<X> sort(List<X>.Comp c){
        List<X> srt = List<X>.create();
        foreach(X x in this)srt = srt.insert(x, c);
        return srt;
    }
    /** Convert this List into an Array, starting at Index 'i' */
    public X[] toArray(X[] arr){
        int i = 0;
        foreach(X x in this)arr[i++] = x;
        return arr;
    }
    
    /** Support foreach(over List elements */
    public IEnumerator<X> GetEnumerator(){ return new ListEnum<X>(this); }
    /** Support foreach(over List elements */
    System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator(){ return GetEnumerator(); }

    class ListEnum<X> : IEnumerator<X>{
        List<X> list;
        List<X> curr;
        public ListEnum(List<X> l){ list = l; curr = null; }
        void check(string op){
            if(curr == null || curr.isEmpty())
                throw new InvalidOperationException(op);
        }
        public X Current{ get{ check("Current"); return curr.top(); } }
        object System.Collections.IEnumerator.Current{ get{ return Current; } }
        public void Dispose(){ list = curr = null; }
        public bool MoveNext(){
            if(curr == null)curr = list;
            else{
                check("MoveNext");
                curr = curr.pop();
            }
            return !curr.isEmpty();
        }
        public void Reset(){ curr = null; }
    }

    /** Zip two lists (this, and 'l') into a single list, one element at a time */
    public List<Z> zip<Y,Z>(List<X>.Zip<Y,Z> z, List<Y> ys){
        List<Z> zs = List<Z>.create();
        List<X> xs = this;
        while(!xs.isEmpty() && !ys.isEmpty()){
            zs = zs.push(z.zip(xs.top(), ys.top()));
            xs = xs.pop();
            ys = ys.pop();
        }
        return zs.reverse();
    }
    /** Build a list from the numbers 0..(len-1) */
    public static List<X> buildlist(List<X>.Build b, int len){
        List<X> lst = List<X>.create();
        while(len-- > 0)lst = lst.push(b.build(len));
        return lst;
    }
    /** Replace the element at index 'i' with 's' */
    public List<X> replace(int i, X s){
        List<X> front = List<X>.create(), back = this;
        while(!back.isEmpty()){
            X x = back.top();
            back = back.pop();
            if(i-- == 0)return back.push(s).revpush(front);
            front = front.push(x);
        }
        throw new Exception("Bad Replace");
    }
    /** Replace the first occurence of 't' with 's' */
    public List<X> replace(X t, X s){ return replace(new EqualComp().curry(t),s); }
    /** Replace the first matching X with 's' */
    public List<X> replace(List<X>.Pred p, X s){
        List<X> front = List<X>.create(), back = this;
        while(!back.isEmpty()){
            X x = back.top();
            back = back.pop();
            if(p.huh(x))return back.push(s).revpush(front);
            front = front.push(x);
        }
        throw new Exception("Bad Replace");
    }
    /** Replace all occurences of 't' with 's' */
    public List<X> replaceAll(X t, X s){ return replaceAll(new EqualComp().curry(t),s); }
    /** Replace all matching Xs with 't' */
    public List<X> replaceAll(List<X>.Pred p, X x){
        List<X> front = List<X>.create(), back = this;
        while(!back.isEmpty()){
            X xx = back.top();
            back = back.pop();
            front = front.push(p.huh(xx)?x:xx);
        }
        return front.reverse();
    }
    
    /** Equals for Lists... */
    public abstract bool Equals(Object o);
    /** HashCode for Lists... */
    public abstract int GetHashCode();
    
    private class FB<X>{
        public List<X> front, back;
        public FB(List<X> f, List<X> b){ front = f; back = b; }
        public override String ToString(){ return front+"::"+back; }
    } 
    private FB<X> frontBack(int i){
        List<X> front = List<X>.create(),
                back = this;
        while(i-- > 0 && !back.isEmpty()){
            front = front.push(back.top());
            back = back.pop();
        }
        return new FB<X>(front,back);
    }
    private static List<X> merge(List<X> a, List<X> b, List<X>.Comp c){
        List<X> ret = List<X>.create();
        foreach(X x in a){
            while(!b.isEmpty() && c.comp(b.top(), x)){
                ret = ret.push(b.top());
                b = b.pop();
            }
            ret = ret.push(x);
        }
        foreach(X x in b)ret = ret.push(x);
        return ret.reverse();
    }
    public List<X> mergeSort(List<X>.Comp c){
        if(length() < 2)return this;
        FB<X> fb = frontBack(length()/2); 
        return merge(fb.front.mergeSort(c),fb.back.mergeSort(c),c);
    }
    
    
    /** Convert this List into a java.util.List */
    public System.Collections.Generic.List<X> toCSList(){
        System.Collections.Generic.List<X> l = new System.Collections.Generic.List<X>();
        foreach(X x in this)l.Add(x);
        return l;
    }
    
    /** Return a sublist, starting at index i, with length k */
    public List<X> sublist(int i, int k){
        if(length() < i+k)
            return List<X>.create();
        return pop(i).reverse(k).reverse();
    }
    
    /** Shuffle the elements of this list randomly */
    public List<X> shuffle(){
        Random rand = new Random();
        List<X> lst = List<X>.create();
        foreach(X x in this)
            lst = lst.add(x, rand.Next(lst.length()+1));
        return lst;
    }
}

}