(* file: lexer.mll *)
{
  open Parse 
  let depth = ref 0 
  let inc r = r := !r + 1
  let dec r = r := !r - 1
}
let white = ' ' | '\t' | '\n' | '\r'
let digit = ['0'-'9']
let identifier = ['a'-'z' 'A'-'Z'] (['a'-'z' 'A'-'Z' '0'-'9' '_'])*

rule token = parse
  | white	                { token lexbuf }
  | "(*"                        { inc depth ; comment lexbuf }
  | digit+ as num               { INT (int_of_string num) }
  | "let"                       { LET }
  | "in"                        { IN }
  | identifier as text          { VAR (text) }
  | '('                         { LPAREN }
  | ')'                         { RPAREN }
  | '+'		                { PLUS }
  | '-'		                { MINUS }
  | '*'		                { TIMES }
  | '='                         { EQ }
  | '/'		                { DIV }
  | ';'                         { SEMI }
  | eof		                { EOF }

and comment = parse
  | "*)"                        { dec depth ; if !depth = 0 then token lexbuf else comment lexbuf }
  | "(*"                        { inc depth ; comment lexbuf }
  | eof                         { raise (Failure "missing comment terminator") }
  | _                           { comment lexbuf }
