/*
 * utils.c
 *
 * This code is in the public domain. I would appreciate bug reports and
 * enhancements.
 *
 * Duncan S Wong  <swong@ieee.org>
 *
 * Dec 15, 2000 - Initial Version
 */

#include <PalmOS.h>
#include "TestbedRsc.h"		// application resource defines
#include "utils.h"

static char hs[] = "0123456789ABCDEF";

static FieldPtr SetFieldTextFromHandle(UInt16 fieldID, MemHandle txtH);


/***********************************************************************
 *
 * FUNCTION:    GetObjectPtr
 *
 * DESCRIPTION: This routine returns a pointer to an object in the active form.
 *
 * PARAMETERS:  objectID - id of the object
 *
 * RETURNED:    pointer to the object's data structure
 *
 ***********************************************************************/
MemPtr GetObjectPtr(UInt16 objectID)
{
FormPtr frm;
	
	frm = FrmGetActiveForm();
	return(FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, objectID)));
}


/***********************************************************************
 * convert strlen bytes of str to i
 *
 * RETURNED: 0 for no error
 **********************************************************************/
Err myStrNToUL(UInt32 *i, Char *str, UInt16 strlen)
{
long tmp;

	*i = 0;
	if(!strlen) return 0;
	if(strlen > 4) return 1;

	if(strlen == 4) {
		// Hey! all commented statements doesn't work!
		//*i = *(str++);
		*i = (unsigned char) *(str++);
		*i = *i << 24;
		//tmp = *(str++);
	}
	if(strlen >= 3) {
		tmp = (unsigned char) *(str++);
		tmp = tmp << 16;
		*i += tmp;
		//*i += *(str++) << 16;
	}
	if(strlen >= 2) {
		//*i += *(str++) << 8;
		tmp = (unsigned char) *(str++);
		tmp = tmp << 8;
		*i += tmp;
	}
	if(strlen >=1) {
		tmp = (unsigned char) *str;
		*i += tmp;
	}

	return 0;
}


static FieldPtr SetFieldTextFromHandle(UInt16 fieldID, MemHandle txtH)
{
MemHandle oldTxtH;
FieldPtr fldP;

	fldP = GetObjectPtr(fieldID);
	oldTxtH = FldGetTextHandle(fldP);
	
	// set the field's text to the new text.
	FldSetTextHandle(fldP, txtH);
	FldDrawField(fldP);
	
	//free the old handle AFTER we call FldSetTextHandle().
	if (oldTxtH) MemHandleFree(oldTxtH);
	
	return fldP;
}


FieldPtr SetFieldTextFromStr(UInt16 fieldID, Char *strP)
{
MemHandle txtH;

  if(strP == NULL) return(NULL);

	// get some space in which to stash the string
	txtH = MemHandleNew(StrLen(strP) + 1);
	if( txtH == NULL) {
    ErrDisplay("Memory allocation error in SetFieldTextFromStr");
    return NULL;
  }
		
	// copy the string to the handle
	StrCopy(MemHandleLock(txtH), strP);
	MemHandleUnlock(txtH);
	
	// set the field to the handle
	return SetFieldTextFromHandle(fieldID, txtH);
}


// do a hexdump of len bytes of in_str (output as the return of the function)
Char *hexdump(UInt8 *in_str, UInt32 len)
{
long i;
Char *outP = NULL;

	outP = MemPtrNew(len*3+1);
	ErrFatalDisplayIf(outP==NULL, "Memory allocation error in hexdump.");

	for(i=0; i<len; i++) {
		outP[3*i]      = hs[in_str[i]>>4];
		outP[3*i+1]    = hs[in_str[i]&15];
		outP[3*i+2]    = ' ';
	}
	outP[3*i] = '\0';

  return(outP);
}


/***********************************************************************
 *
 * FUNCTION:    RomVersionCompatible
 *
 * DESCRIPTION: Check that the ROM version meets your
 *              minimum requirement.  Warn if the app was switched to.
 *
 * PARAMETERS:  requiredVersion - minimum rom version required
 *                                (see sysFtrNumROMVersion in SystemMgr.h 
 *                                for format)
 *              launchFlags     - flags indicating how the application was
 *											 launched.  A warning is displayed only if
 *                                these flags indicate that the app is 
 *											 launched normally.
 *
 * RETURNED:    zero if rom is compatible else an error code
 *                             
 ***********************************************************************/
Err RomVersionCompatible (UInt32 requiredVersion, UInt16 launchFlags)
{
UInt32 romVersion;
	
	// See if we're on in minimum required version of the ROM or later.
	// The system records the version number in a feature.  A feature is a
	// piece of information which can be looked up by a creator and feature
	// number.
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (romVersion < requiredVersion) {
		// If the user launched the app from the launcher, explain
		// why the app shouldn't run.  If the app was contacted for something
		// else, like it was asked to find a string by the system find, then
		// don't bother the user with a warning dialog.  These flags tell how
		// the app was launched to decided if a warning should be displayed.
		if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
		{
			FrmAlert (RomIncompatibleAlert);
		
			// Pilot 1.0 will continuously relaunch this app unless we switch to 
			// another safe one.  The sysFileCDefaultApp is considered "safe".
			if (romVersion < 0x02000000)
				AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
		}
		
		return (sysErrRomIncompatible);
	}

	return 0;
}


/*
 * Copy a block of data from one memory buffer to another
 *
 * Note: This routine supports overlapping memory blocks
 */
void MemCopy(UInt8 *to, UInt8 *from, UInt16 len)
{
int i;
UInt8 *strP;

  strP = MemPtrNew(len);
  for (i=0; i<len; i++)
    strP[i] = from[i];
  for (i=0; i<len; i++)
    to[i] = strP[i];
  MemPtrFree(strP);
}
