/*
 * mp_prime.c
 *
 * This code is in the public domain. I would appreciate bug reports and
 * enhancements.
 *
 * Duncan S Wong <swong@ieee.org>
 *
 * Feb 11, 2001 - Initial Version
 */
#include <PalmOS.h>
#include "mp.h"
#include "mp_priv.h"

static Int16 MP_isprime(DIGIT n);
static Int16 MP_millerrabin(INT *n, UInt16 reps, MP_CTX *ctx);

// If it returns 1, then n is 'probably' prime;
// if it returns 2, then n is surely prime;
// if it returns 3, then n is not prime;
// if it returns 0, then something's going wrong.
//
// If n is small, the function uses the trivial division,
// i.e. try to divide by 3, 5, 7,...; otherwise first try to divide n
// by small primes; and then run the Miller-Rabin primality test for
// (UInt16) reps times.
//
// This is a simplified implementation of the probabilistic primality test
// described in Knuth's Seminumerical Algorithms (Vol II).
//
// The probability of a false positive is a/(a + 2^(2m+1) where
// a = |n|-2 and m is the number of times the Miller-Rabin primality test
// has been run. (Check Stinson's Cryptography book pp.135--138 for details)
// In practice, Stinson recommends m be 50 or 100 while Knuth recommends
// m = 25 for reasonable confidence.

Int16 MP_probab_prime(INT *n, UInt16 reps, MP_CTX *ctx)
{
int i;
int ret=0;

	if( (n->top == 1) && (n->d[0] <= (DIGIT)3) )
		return(2);  // n is 1, 2 (even prime) or 3
	if(!MP_is_odd(n)) return(3);  // If n is now even, it's not a prime

  // Handle small and negative n
	if( (n->top == 1) && (n->d[0] <= 1000000L)) {
		ret = MP_isprime(n->d[0]);
		return(ret);
	}

	{  // Check if n has small prime factors
		INT *m=NULL, *t1;

		// the product of the first 61 primes (from 3 to 293)
		m = MP_ascii2bn("654ef8846a605086d8f809a4a47bf677d1af73120e8341e6ff4d9c83af3fd84d34bc238a091bfc23438da95794510029d245");
		if(m == NULL) return(ret);
		t1 = ctx->t[ctx->tos];
		ctx->tos++;
		ret = MP_gcd(t1, n, m, ctx);
		ctx->tos--;
		if(!ret) return(ret);
		if(!MP_is_one(t1)) return(3);   // gcd(n,m) > 1 --- not relatively prime
		MP_free(m);
	}

  // Perform a number of Miller-Rabin tests
	if(n->neg) { i=1; n->neg = 0; }
	else i=0;
  ret = MP_millerrabin (n, reps, ctx);
	if(i) n->neg = 1;

	return(ret);
}


// Trivial Divison
// try to divide n by 3, 5, ... up to the upper ceiling of sqrt(n)
// Returns 2 if n is prime;
//         3 if n is composite.
static Int16 MP_isprime (DIGIT n)
{
DIGIT q, r, d;

	for (d = 3, r = 1; r != 0; d += 2) {
		q = n / d;
		r = n - q * d;
		if (q < d) return 2;
	}
  return 3;
}



// Miller-Rabin primality test for an odd integer n
// Check Stinson's book Figure 4.9 on pp.137 for details
// Note: - a should be a random integer between 1 and n-1
//         but for simplicity, I set a to 2 and increases it
//         by one for each run
// If it returns 1, then n is 'probably' prime;
// if it returns 3, then n is not prime;
// if it returns 0, then something's going wrong.
static Int16 MP_millerrabin (INT *n, UInt16 reps, MP_CTX *ctx)
{
INT *a, *m, *b, *u, *t1;
UInt16 i;
UInt32 num_bits, k, j;
MP_MONT_CTX *mont=NULL;
int ret=0;

	mont = MP_MONT_CTX_new();
	if(mont==NULL) return(ret);
	if(!MP_MONT_CTX_set(mont, n, ctx)) goto err2;

	a = ctx->t[ctx->tos];
	m = ctx->t[ctx->tos+1];
	b = ctx->t[ctx->tos+2];
	u = ctx->t[ctx->tos+3];
	t1 = ctx->t[ctx->tos+4];
	ctx->tos+=5;

	a->d[0] = (DIGIT)2;
	MP_one(t1);
	MP_sub(u, n, t1);

	num_bits = MP_num_bits(u);
	for(k=0; k<num_bits; k++)
		if(MP_is_bit_set(u, k)) break;
	if(!MP_rshift(m, u, k)) goto err;

	ret = 1;
	for(i=0; i<reps; i++) { // run Miller-Rabin primality test for reps times
		MP_mont_exp(b, a, m, mont, ctx);
		if(!MP_is_one(b)) {
			for(j=0; j<k; j++)
				if(MP_cmp(b, u)) {
					if(!MP_mod_mul(t1, b, b, n, ctx)) { ret = 0; goto err; }
					if(MP_copy(b, t1) == NULL) { ret = 0; goto err; }
				} else {
					break;
				}
			if(j>=k) {
				ret = 3;
				break;
			}
		}
		MP_one(t1);
		MP_add(a, a, t1);
	}

err:
	ctx->tos-=5;
err2:
	if(mont!=NULL) MP_MONT_CTX_free(mont);
	return(ret);
}